<?php
/**
 * Functions and definitions
 *
 * @package WordPress
 * @subpackage Orin
 * @since Orin 1.0
 */

/**
 * TGM Plugin Activation
 *
 * @since Orin 1.0
 */
require_once get_theme_file_path( '/assets/class-tgm-plugin-activation.php' );
require_once get_theme_file_path( '/assets/plugin-activation.php' );


/**
 * Theme meta boxes
 *
 * @since Orin 1.0
 */
require_once get_theme_file_path( '/assets/theme-meta-boxes.php' );


/**
 * Set up the content width value
 *
 * @since Orin 1.0
 */
if ( ! isset( $content_width ) ) {
	$content_width = 1000;
}


/**
 * Sets up theme defaults and registers support for various WordPress features
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_setup' ) ) {
	function orin_setup() {

		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 */
		load_theme_textdomain( 'orin', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		// Register menus
		register_nav_menus( array(
			'orin_main_menu'	=> esc_html__( 'Main menu (Header)', 'orin' ),
			'orin_footer_menu'	=> esc_html__( 'Footer menu', 'orin' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Enable support for Post Formats.
		add_theme_support( 'post-formats', array(
			'image',
			'gallery',
			'video',
			'audio',
		) );

		// Enable support for Custom logo
		add_theme_support( 'custom-logo', array(
			'height'		=> 50,
			'width'			=> 148,
			'flex-height'	=> true,
			'flex-width'	=> true,
		) );

		// Add styles for TinyMCE editor (fonts and editor-style.css)
		$editor_fonts_url = orin_google_fonts_url_for_editors();
		$editor_style_args = array(
			esc_url_raw( $editor_fonts_url ),
			'css/editor-style.css',
		);
		add_theme_support( 'editor-styles' );
		add_editor_style( $editor_style_args );

		// Block editor: Add support for wide and full alignment
		add_theme_support( 'align-wide' );

		// Block editor: Add support for responsive embeds
		add_theme_support( 'responsive-embeds' );

		// Block editor: Add support for custom line height
		add_theme_support( 'custom-line-height' );

		// Block editor: New color palette (9 colors)
		add_theme_support( 'editor-color-palette', array(
			array(
				'name'	=> esc_attr__( 'Black', 'orin' ),
				'slug'	=> 'black',
				'color'	=> '#000000',
			),
			array(
				'name'	=> esc_attr__( 'Light Gray', 'orin' ),
				'slug'	=> 'light-gray',
				'color'	=> '#8f8f9c',
			),
			array(
				'name'	=> esc_attr__( 'White', 'orin' ),
				'slug'	=> 'white',
				'color'	=> '#ffffff',
			),
			array(
				'name'	=> esc_attr__( 'Green', 'orin' ),
				'slug'	=> 'green',
				'color'	=> '#6fff80',
			),
			array(
				'name'	=> esc_attr__( 'Blue', 'orin' ),
				'slug'	=> 'blue',
				'color'	=> '#6f6fff',
			),
			array(
				'name'	=> esc_attr__( 'Purple', 'orin' ),
				'slug'	=> 'purple',
				'color'	=> '#ff6ff6',
			),
			array(
				'name'	=> esc_attr__( 'Red', 'orin' ),
				'slug'	=> 'red',
				'color'	=> '#ff6f89',
			),
			array(
				'name'	=> esc_attr__( 'Orange', 'orin' ),
				'slug'	=> 'orange',
				'color'	=> '#ffb06f',
			),
			array(
				'name'	=> esc_attr__( 'Yellow', 'orin' ),
				'slug'	=> 'yellow',
				'color'	=> '#fffb6f',
			),
		) );

		// Enable support for Post Thumbnails
		add_theme_support( 'post-thumbnails' );
		set_post_thumbnail_size( 1920, 1280, true ); // 1920x1280px, cropped (image size identifier = 'post-thumbnail')

		// New image sizes
		add_image_size( 'orin-200-200-crop', 200, 200, true ); // thumbnail for widgets (200x200px, cropped)
		add_image_size( 'orin-900-600-crop', 900, 600, true ); // thumbnail for related posts (900x600px, cropped)

	}
}
add_action( 'after_setup_theme', 'orin_setup' );


/**
 * The function returns the URL of all Google fonts that are used in this theme
 *
 * This theme uses Google fonts version 2
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_google_fonts_url' ) ) {
	function orin_google_fonts_url() {

		/*
		 * This theme uses sans-serif fonts by default.
		 *
		 * If switching between font types is enabled in the theme settings, then the theme loads sans-serif and serif fonts.
		 */
		$switch_between_font_types = get_theme_mod( 'orin_enable_font_switch', 0 ); // 1 or 0

		// Get fonts url:
		if ( $switch_between_font_types ) {

			/*
			 * Switching between font types is enabled in the theme settings, so we load sans-serif and serif fonts.
			 * We also load only those weights that are used in the theme.
			 *
			 * Sans-serif fonts:
			 * - Main font: Open Sans (styles: 400, 400 italic, 700, 700 italic);
			 * - Additional font (headings h1...h6, main menu links, search form): Poppins (styles: 300, 500, 600, 600 italic, 700, 700 italic);
			 * - Metadata and buttons: Source Sans Pro (styles: 400, 400 italic, 600);
			 * - Text logo and quotes: Yeseva One (style: 400).
			 * Serif fonts:
			 * - Main font (headings and main content): Lora (styles: 400, 400 italic, 600, 700, 700 italic);
			 * - Metadata and buttons: Noto Serif (styles: 400, 400 italic, 700).
			 */
			$fonts_url = '//fonts.googleapis.com/css2?family=Lora:ital,wght@0,400;0,600;0,700;1,400;1,700&family=Noto+Serif:ital,wght@0,400;0,700;1,400&family=Open+Sans:ital,wght@0,400;0,700;1,400;1,700&family=Poppins:ital,wght@0,300;0,500;0,600;0,700;1,600;1,700&family=Source+Sans+Pro:ital,wght@0,400;0,600;1,400&family=Yeseva+One&display=swap';

		} else {

			/*
			 * Switching between font types is disabled, so we load sans-serif or serif fonts depending on the selected fonts.
			 * We also load only those weights that are used in the theme.
			 */

			$default_font_type = get_theme_mod( 'orin_default_font_type', 'sans-serif' ); // sans-serif or serif
			if ( 'sans-serif' === $default_font_type ) {

				/*
				 * Sans-serif fonts:
				 * - Main font: Open Sans (styles: 400, 400 italic, 700, 700 italic);
				 * - Additional font (headings h1...h6, main menu links, search form): Poppins (styles: 300, 500, 600, 600 italic, 700, 700 italic);
				 * - Metadata and buttons: Source Sans Pro (styles: 400, 400 italic, 600);
				 * - Text logo and quotes: Yeseva One (style: 400).
				 */
				$fonts_url = '//fonts.googleapis.com/css2?family=Open+Sans:ital,wght@0,400;0,700;1,400;1,700&family=Poppins:ital,wght@0,300;0,500;0,600;0,700;1,600;1,700&family=Source+Sans+Pro:ital,wght@0,400;0,600;1,400&family=Yeseva+One&display=swap';

			} else {

				/*
				 * Serif fonts:
				 * - Main font (headings and main content): Lora (styles: 400, 400 italic, 600, 700, 700 italic);
				 * - Metadata and buttons: Noto Serif (styles: 400, 400 italic, 700);
				 * + Text logo and quotes: Yeseva One (style: 400).
				 */
				$fonts_url = '//fonts.googleapis.com/css2?family=Lora:ital,wght@0,400;0,600;0,700;1,400;1,700&family=Noto+Serif:ital,wght@0,400;0,700;1,400&family=Yeseva+One&display=swap';

			}

		}

		// Return URL of fonts. This URL will be escaped right before use.
		return $fonts_url;

	}
}


/**
 * Enqueue styles for the block editor
 *
 * @since Orin 1.2
 */
if ( ! function_exists( 'orin_block_assets' ) ) {
	function orin_block_assets() {
		// We need these styles only on the admin side:
		if ( is_admin() ) {

			// Google fonts
			$fonts_url = orin_google_fonts_url_for_editors();
			wp_enqueue_style( 'orin-gutenberg-fonts', esc_url_raw( $fonts_url ), array(), null );

			// Styles for the editor
			wp_enqueue_style( 'orin-gutenberg-editor-style', get_template_directory_uri() . '/css/gutenberg-editor-style.css', array(), '1.2.0' );

		}
	}
}
add_action( 'enqueue_block_assets', 'orin_block_assets' );


/**
 * Enqueue all styles and scripts for the theme
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_styles_scripts' ) ) {
	function orin_styles_scripts() {

		/**
		 * CSS Styles:
		 */

		// Google fonts
		$fonts_url = orin_google_fonts_url();
		wp_enqueue_style( 'orin-fonts', esc_url_raw( $fonts_url ), array(), null );

		// bootstrap
		wp_enqueue_style( 'bootstrap', get_template_directory_uri() . '/css/bootstrap.min.css', array(), '3.4.1' );
		wp_enqueue_style( 'bootstrap-theme', get_template_directory_uri() . '/css/bootstrap-theme.min.css', array(), '3.4.1' );

		// ie10 viewport bug workaround
		wp_enqueue_style( 'orin-ie10-viewport-bug-workaround', get_template_directory_uri() . '/css/ie10-viewport-bug-workaround.css', array(), '3.4.1' );

		// font awesome (icons)
		wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/assets/fontawesome/css/all.min.css', array(), '5.15.4' );

		// tiny slider
		wp_enqueue_style( 'tiny-slider', get_template_directory_uri() . '/assets/tiny-slider/tiny-slider.css', array(), '2.9.3' );

		// magnific popup
		wp_enqueue_style( 'magnific-popup', get_template_directory_uri() . '/assets/magnific-popup/magnific-popup.css', array(), '1.1.0' );

		// main stylesheet
		wp_enqueue_style( 'orin-style', get_stylesheet_directory_uri() . '/style.css', array(), '1.2.0' );

		/**
		 * Scripts:
		 */

		// html5shiv.js (for IE)
		wp_enqueue_script( 'html5shiv', get_template_directory_uri() . '/js/html5shiv.min.js', array(), '3.7.3', false );
		wp_script_add_data( 'html5shiv', 'conditional', 'lt IE 9' );

		// respond.js (for IE)
		wp_enqueue_script( 'respond', get_template_directory_uri() . '/js/respond.min.js', array(), '1.4.2', false );
		wp_script_add_data( 'respond', 'conditional', 'lt IE 9' );

		// tiny slider: IE8 polyfills (for IE)
		wp_enqueue_script( 'tiny-slider-helper-ie8', get_template_directory_uri() . '/assets/tiny-slider/min/tiny-slider.helper.ie8.js', array(), '2.9.3', false );
		wp_script_add_data( 'tiny-slider-helper-ie8', 'conditional', 'lt IE 9' );

		// bootstrap
		wp_enqueue_script( 'bootstrap', get_template_directory_uri() . '/js/bootstrap.min.js', array( 'jquery' ), '3.4.1', true );

		// superfish (drop-down menu)
		wp_enqueue_script( 'superfish', get_template_directory_uri() . '/js/superfish.min.js', array( 'jquery' ), '1.7.10', true );

		// masonry
		if ( ! is_singular() ) {
			wp_enqueue_script( 'jquery-masonry' );
		}

		// tiny-slider
		wp_enqueue_script( 'tiny-slider', get_template_directory_uri() . '/assets/tiny-slider/min/tiny-slider.js', array(), '2.9.3', true );

		// magnific popup
		wp_enqueue_script( 'jquery-magnific-popup', get_template_directory_uri() . '/assets/magnific-popup/jquery.magnific-popup.min.js', array( 'jquery' ), '1.1.0', true );

		// ie10 viewport bug workaround
		wp_enqueue_script( 'orin-ie10-viewport-bug-workaround', get_template_directory_uri() . '/js/ie10-viewport-bug-workaround.js', array(), '3.4.1', true );

		// orin theme js
		wp_enqueue_script( 'orin-theme', get_template_directory_uri() . '/js/orin-theme.js', array( 'jquery' ), '1.0.2', true );

		// comments
		if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
			wp_enqueue_script( 'comment-reply' );
		}

		/**
		 * Data for the main script file (orin-theme.js; id: orin-theme):
		 */

		// page type: single page or not
		$isSingular_data = ( is_singular() ) ? 'true' : 'false';

		// switching between font types: enable / disable
		$enable_font_switch = get_theme_mod( 'orin_enable_font_switch', 0 ); // 1 or 0
		$fontSwitch_data = ( $enable_font_switch ) ? 'on' : 'off';

		// color switch: enable / disable
		$enable_color_switch = get_theme_mod( 'orin_enable_color_switch', 0 ); // 1 or 0
		$colorSwitch_data = ( $enable_color_switch ) ? 'on' : 'off';

		// search icon and dropdown search form: show / hide
		$show_search = get_theme_mod( 'orin_show_search', 1 ); // 1 or 0
		$dropdownSearch_data = ( $show_search ) ? 'on' : 'off';

		// hidden sidebar: enable / disable
		$enable_hidden_sidebar = is_active_sidebar( 'orin_sidebar' ) && get_theme_mod( 'orin_show_sidebar_button', 1 ); // true or false
		$hiddenSidebar_data = ( $enable_hidden_sidebar ) ? 'on' : 'off';

		// logo type: text or image
		$logoType_data = get_theme_mod( 'orin_logo_type', 'text' );

		// homepage slider: slider options
		// 1: rewind
		$homepage_slider_enable_rewind = get_theme_mod( 'orin_slider_enable_rewind', 0 ); // 1 or 0
		$homepageSliderRewind_data = ( $homepage_slider_enable_rewind ) ? 'on' : 'off';
		// 2: transition type for slides
		$homepageSliderTransition_data = get_theme_mod( 'orin_slider_transition_type', 'slide_vertical' ); // 'fade', 'slide_horizontal', or 'slide_vertical'
		// 3: speed of the slide animation
		$homepageSliderAnimationSpeed_data = get_theme_mod( 'orin_slider_animation_speed', 600 );
		if ( '' === $homepageSliderAnimationSpeed_data ) {
			$homepageSliderAnimationSpeed_data = 600;
		}
		// 4: navigation buttons (next/previous)
		$homepage_slider_show_controls = get_theme_mod( 'orin_slider_show_controls', 1 ); // 1 or 0
		$homepageSliderControls_data = ( $homepage_slider_show_controls ) ? 'on' : 'off';
		// 5: pagination (dots)
		$homepage_slider_show_nav = get_theme_mod( 'orin_slider_show_nav', 1 ); // 1 or 0
		$homepageSliderNav_data = ( $homepage_slider_show_nav ) ? 'on' : 'off';

		// blog posts: gallery format slider
		// 1: rewind
		$gallery_slider_enable_rewind = get_theme_mod( 'orin_gallery_slider_enable_rewind', 0 ); // 1 or 0
		$gallerySliderRewind_data = ( $gallery_slider_enable_rewind ) ? 'on' : 'off';
		// 2: transition type for slides
		$gallerySliderTransition_data = get_theme_mod( 'orin_gallery_slider_transition_type', 'slide_vertical' ); // 'fade', 'slide_horizontal', or 'slide_vertical'
		// 3: speed of the slide animation
		$gallerySliderAnimationSpeed_data = get_theme_mod( 'orin_gallery_slider_animation_speed', 600 );
		if ( '' === $gallerySliderAnimationSpeed_data ) {
			$gallerySliderAnimationSpeed_data = 600;
		}

		// cookies for storing the current font type: enable or disable
		$font_switch_enable_cookies = get_theme_mod( 'orin_font_switch_enable_cookies', 1 ); // 1 or 0
		$fontSwitchCookies_data = ( $font_switch_enable_cookies ) ? 'on' : 'off';

		// cookies for storing the current theme style: enable or disable
		$color_switch_enable_cookies = get_theme_mod( 'orin_color_switch_enable_cookies', 1 ); // 1 or 0
		$colorSwitchCookies_data = ( $color_switch_enable_cookies ) ? 'on' : 'off';

		// back to top: show / hide
		$show_back_to_top = get_theme_mod( 'orin_show_back_to_top', 1 ); // 1 or 0
		$backToTop_data = ( $show_back_to_top ) ? 'on' : 'off';

		// data array
		$orinData_array = array(
			'isSingular'					=> $isSingular_data,
			'fontSwitch'					=> $fontSwitch_data,
			'colorSwitch'					=> $colorSwitch_data,
			'dropdownSearch'				=> $dropdownSearch_data,
			'hiddenSidebar'					=> $hiddenSidebar_data,
			'logoType'						=> $logoType_data,
			'homepageSliderRewind'			=> $homepageSliderRewind_data,
			'homepageSliderTransition'		=> $homepageSliderTransition_data,
			'homepageSliderAnimationSpeed'	=> $homepageSliderAnimationSpeed_data,
			'homepageSliderControls'		=> $homepageSliderControls_data,
			'homepageSliderNav'				=> $homepageSliderNav_data,
			'gallerySliderRewind'			=> $gallerySliderRewind_data,
			'gallerySliderTransition'		=> $gallerySliderTransition_data,
			'gallerySliderAnimationSpeed'	=> $gallerySliderAnimationSpeed_data,
			'fontSwitchCookies'				=> $fontSwitchCookies_data,
			'colorSwitchCookies'			=> $colorSwitchCookies_data,
			'backToTop'						=> $backToTop_data,
		);
		// add this data to the main script file (orin-theme.js; id: orin-theme)
		wp_localize_script( 'orin-theme', 'orinData', $orinData_array );

	}
}
add_action( 'wp_enqueue_scripts', 'orin_styles_scripts' );


/**
 * Enqueue styles for the WordPress customizer (Admin side)
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_customizer_styles' ) ) {
	function orin_customizer_styles() {
		// customizer style
		wp_enqueue_style( 'orin-customizer-style', get_template_directory_uri() . '/css/customizer-style.css', array(), '1.2.0' );
	}
}
add_action( 'customize_controls_enqueue_scripts', 'orin_customizer_styles' );


/**
 * Enqueue scripts for the block editor (Admin side)
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_block_editor_assets' ) ) {
	function orin_block_editor_assets() {
		// show/hide meta boxes
		wp_enqueue_script( 'orin-gutenberg-meta-boxes', get_template_directory_uri() . '/js/orin-gutenberg-meta-boxes.js', array( 'jquery' ), '1.0.1', true );
	}
}
add_action( 'enqueue_block_editor_assets', 'orin_block_editor_assets' );


/**
 * Enqueue scripts for post editors (Classic editor and Block editor; Admin side)
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_admin_scripts' ) ) {
	function orin_admin_scripts( $hook ) {
		// add scripts only to the post creation/editing page
		if ( 'post-new.php' !== $hook && 'post.php' !== $hook ) {
			return; // stop the function
		}
		// meta boxes
		wp_enqueue_script( 'orin-meta-boxes', get_template_directory_uri() . '/js/orin-meta-boxes.js', array( 'jquery' ), '1.0.0', true );
	}
}
add_action( 'admin_enqueue_scripts', 'orin_admin_scripts' );


/**
 * Google fonts for WordPress editors (Classic editor and Block editor)
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_google_fonts_url_for_editors' ) ) {
	function orin_google_fonts_url_for_editors() {

		/*
		 * The Classic Editor and the Block Editor use sans-serif fonts.
		 * We also load only those weights that are used in the editors.
		 * The following fonts are used:
		 * - Open Sans (styles: 400, 400 italic, 700, 700 italic);
		 * - Poppins (styles: 600, 600 italic, 700, 700 italic);
		 * - Source Sans Pro (styles: 400, 600);
		 * - Yeseva One (style: 400).
		 */
		return 'https://fonts.googleapis.com/css2?family=Open+Sans:ital,wght@0,400;0,700;1,400;1,700&family=Poppins:ital,wght@0,600;0,700;1,600;1,700&family=Source+Sans+Pro:wght@400;600&family=Yeseva+One&display=swap';

	}
}


/**
 * Add preconnect for Google Fonts
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_resource_hints' ) ) {
	function orin_resource_hints( $urls, $relation_type ) {
		if ( wp_style_is( 'orin-fonts', 'queue' ) && 'preconnect' === $relation_type ) {
			if ( version_compare( $GLOBALS['wp_version'], '4.7-alpha', '>=' ) ) {
				$urls[] = '//fonts.googleapis.com';
				$urls[] = array(
					'href' => '//fonts.gstatic.com',
					'crossorigin',
				);
			} else {
				$urls[] = '//fonts.googleapis.com';
				$urls[] = '//fonts.gstatic.com';
			}
		}

		return $urls;
	}
}
add_filter( 'wp_resource_hints', 'orin_resource_hints', 10, 2 );


/**
 * The function adds the wp_body_open action for backward compatibility to support WordPress versions prior to 5.2
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'wp_body_open' ) ) {
	function wp_body_open() {
		// Triggered after the opening "body" tag
		do_action( 'wp_body_open' );
	}
}


/**
 * The function registers areas for widgets
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_widgets_init' ) ) {
	function orin_widgets_init() {

		// Main sidebar (this sidebar is located in a hidden container on the right side)
		register_sidebar( array(
			'name'			=> esc_html__( 'Sidebar', 'orin' ),
			'id'			=> 'orin_sidebar',
			'description'	=> esc_html__( 'Add widgets here to appear in your hidden sidebar. This area for widgets is located in a hidden container on the right.', 'orin' ),
			'before_widget'	=> '<div id="%1$s" class="bwp-widget %2$s clearfix">',
			'after_widget'	=> '</div>',
			'before_title'	=> '<h3 class="bwp-widget-title">',
			'after_title'	=> '</h3>',
		) );

		// Footer column 1 (left column)
		register_sidebar( array(
			'name'			=> esc_html__( 'Footer: Left Column', 'orin' ),
			'id'			=> 'orin_footer_sidebar_left',
			'description'	=> esc_html__( 'Add widgets here to appear in your footer (column 1: left column).', 'orin' ),
			'before_widget'	=> '<div id="%1$s" class="bwp-widget %2$s clearfix">',
			'after_widget'	=> '</div>',
			'before_title'	=> '<h3 class="bwp-widget-title">',
			'after_title'	=> '</h3>',
		) );

		// Footer column 2 (center column)
		register_sidebar( array(
			'name'			=> esc_html__( 'Footer: Center Column', 'orin' ),
			'id'			=> 'orin_footer_sidebar_center',
			'description'	=> esc_html__( 'Add widgets here to appear in your footer (column 2: center column).', 'orin' ),
			'before_widget'	=> '<div id="%1$s" class="bwp-widget %2$s clearfix">',
			'after_widget'	=> '</div>',
			'before_title'	=> '<h3 class="bwp-widget-title">',
			'after_title'	=> '</h3>',
		) );

		// Footer column 3 (right column)
		register_sidebar( array(
			'name'			=> esc_html__( 'Footer: Right Column', 'orin' ),
			'id'			=> 'orin_footer_sidebar_right',
			'description'	=> esc_html__( 'Add widgets here to appear in your footer (column 3: right column).', 'orin' ),
			'before_widget'	=> '<div id="%1$s" class="bwp-widget %2$s clearfix">',
			'after_widget'	=> '</div>',
			'before_title'	=> '<h3 class="bwp-widget-title">',
			'after_title'	=> '</h3>',
		) );

	}
}
add_action( 'widgets_init', 'orin_widgets_init' );


/**
 * The function returns the current theme style (light style or dark style)
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_get_current_theme_style' ) ) {
	function orin_get_current_theme_style() {
		// default theme style
		$default_theme_style = get_theme_mod( 'orin_default_theme_style', 'light' ); // light or dark

		// if switching between theme styles is enabled, and cookies are used for this, then we get the current theme style:
		$enable_color_switch = get_theme_mod( 'orin_enable_color_switch', 0 ); // 1 or 0
		$color_switch_enable_cookies = get_theme_mod( 'orin_color_switch_enable_cookies', 1 ); // 1 or 0

		if ( $enable_color_switch && $color_switch_enable_cookies ) {

			$current_theme_style = ( ! empty( $_COOKIE['orin_site_style'] ) ) ? $_COOKIE['orin_site_style'] : $default_theme_style;
			return $current_theme_style; // light or dark

		}

		// if color switching is off, return the default theme style
		return $default_theme_style;
	}
}


/**
 * The function returns the current font type (sans-serif or serif fonts)
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_get_current_font_type' ) ) {
	function orin_get_current_font_type() {
		// default font type
		$default_font_type = get_theme_mod( 'orin_default_font_type', 'sans-serif' ); // sans-serif or serif

		// if switching between font types is activated, and cookies are used for this, then we get the current font type:
		$switch_between_font_types = get_theme_mod( 'orin_enable_font_switch', 0 ); // 1 or 0
		$switch_between_font_types_cookies = get_theme_mod( 'orin_font_switch_enable_cookies', 1 ); // 1 or 0

		if ( $switch_between_font_types && $switch_between_font_types_cookies ) {

			$current_font_type = ( ! empty( $_COOKIE['orin_font_type'] ) ) ? $_COOKIE['orin_font_type'] : $default_font_type;
			return $current_font_type; // sans-serif or serif

		}

		// if switching between font types is disabled, then we return the default font type
		return $default_font_type;
	}
}


/**
 * The function adds additional classes to the "body" element
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_custom_body_classes' ) ) {
	function orin_custom_body_classes( $classes ) {
		// add two additional classes
		$classes[] = 'bwp-body';
		$classes[] = 'bwp-sidebar-hidden';
		// class for the dark version of the theme
		$current_theme_style = orin_get_current_theme_style();
		if ( 'dark' === $current_theme_style ) {
			$classes[] = 'bwp-dark-style';
		}
		// class for font type
		$current_font_type = orin_get_current_font_type();
		if ( 'serif' === $current_font_type ) {
			$classes[] = 'bwp-serif-fonts';
		}
		// if there is no site header
		$header_has_menu = has_nav_menu( 'orin_main_menu' ); // true or false
		$show_sidebar_button = is_active_sidebar( 'orin_sidebar' ) && get_theme_mod( 'orin_show_sidebar_button', 1 ); // true or false
		$show_search = get_theme_mod( 'orin_show_search', 1 ); // 1 or 0
		$show_color_switch = get_theme_mod( 'orin_enable_color_switch', 0 ); // 1 or 0
		$show_font_types = get_theme_mod( 'orin_enable_font_switch', 0 ); // 1 or 0
		if ( ! $header_has_menu && ! $show_sidebar_button && ! $show_search && ! $show_color_switch && ! $show_font_types ) {
			$classes[] = 'bwp-no-site-header';
		}
		// class for sticky header
		$enable_sticky_header = get_theme_mod( 'orin_enable_sticky_header', 1 ); // 1 or 0
		if ( $enable_sticky_header ) {
			$classes[] = 'bwp-enable-sticky-header';
		} else {
			$classes[] = 'bwp-disable-sticky-header';
		}

		// return classes
		return $classes;
	}
}
add_filter( 'body_class', 'orin_custom_body_classes' );


/**
 * The function adds additional classes to posts
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_custom_post_classes' ) ) {
	function orin_custom_post_classes( $classes, $class, $post_id ) {
		if ( ! is_singular() ) {
			// Add additional CSS classes for posts on archive pages
			$classes[] = 'bwp-masonry-item';
			$classes[] = 'bwp-col-3';
			// post title:
			$classes[] = ( get_the_title( $post_id ) ) ? 'bwp-post-has-title' : 'bwp-no-title';
			// post size:
			$site_page = get_theme_mod( 'orin_increase_first_post_on_page', 'homepage' ); // 'homepage', 'all-archives', 'no-increase'
			$increase_first_post = false;
			if ( 'homepage' === $site_page ) {
				$increase_first_post = is_home();
			} elseif ( 'all-archives' === $site_page ) {
				$increase_first_post = true;
			}
			// increase the first post on pagination pages or not?
			$increase_first_post_on_pagination = ( get_theme_mod( 'orin_increase_first_post_on_pagination', 0 ) ) ? true : ! is_paged();
			if ( $increase_first_post && $increase_first_post_on_pagination ) {
				global $wp_query;
				if ( 0 === $wp_query->current_post ) {
					$classes[] = 'bwp-width-2x'; // the first post has a double width
				}
			}
		} else {
			// Add additional CSS classes for single posts and pages
			if ( is_single( $post_id ) ) {
				// single post page
				$classes[] = 'bwp-single-post-article';
			} elseif ( is_page( $post_id ) ) {
				// single page
				$classes[] = 'bwp-single-post-article';
				$classes[] = 'bwp-page-article';
			}
		}

		// return classes
		return $classes;
	}
}
add_filter( 'post_class', 'orin_custom_post_classes', 10, 3 );


/**
 * Change excerpt length
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_new_excerpt_length' ) ) {
	function orin_new_excerpt_length( $length ) {
		$new_excerpt_length = get_theme_mod( 'orin_excerpt_length', 25 );

		if ( $new_excerpt_length ) {
			$new_excerpt_length = intval( $new_excerpt_length );
		} else {
			$new_excerpt_length = 25; // default value
		}

		return $new_excerpt_length;
	}
}
add_filter( 'excerpt_length', 'orin_new_excerpt_length' );


/**
 * The function returns true if the post has one of the following attachments: featured image, gallery, video, or audio. Otherwise, the function returns false.
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_post_has_featured_media' ) ) {
	function orin_post_has_featured_media( $blog_post_id, $blog_post_format ) {

		// gallery format
		if ( 'gallery' === $blog_post_format ) {

			// slider: image IDs
			$gallery_images_id = get_post_meta( $blog_post_id, 'orin_mb_gallery', false );
			if ( ! is_array( $gallery_images_id ) ) {
				$gallery_images_id = (array) $gallery_images_id;
			}
			// if $gallery_images_id is not empty
			if ( ! empty( $gallery_images_id ) && $gallery_images_id[0] ) {
				return true;
			} elseif ( has_post_thumbnail( $blog_post_id ) ) {
				return true;
			} else {
				return false;
			}

		} elseif ( 'video' === $blog_post_format ) {
			// video format

			$video_url = get_post_meta( $blog_post_id, 'orin_mb_video_url', true );
			if ( $video_url ) {
				return true;
			} elseif ( has_post_thumbnail( $blog_post_id ) ) {
				return true;
			} else {
				return false;
			}

		} elseif ( 'audio' === $blog_post_format ) {
			// audio format

			$audio_url = get_post_meta( $blog_post_id, 'orin_mb_audio_url', true );
			if ( $audio_url ) {
				return true;
			} elseif ( has_post_thumbnail( $blog_post_id ) ) {
				return true;
			} else {
				return false;
			}

		} else {
			// other formats

			if ( has_post_thumbnail( $blog_post_id ) ) {
				return true;
			} else {
				return false;
			}

		}

	}
}


/**
 * The function displays a custom logo (image or text)
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_show_custom_logo' ) ) {
	function orin_show_custom_logo() {
		// logo type: text or image
		$logo_type = get_theme_mod( 'orin_logo_type', 'text' );
		if ( 'text' === $logo_type ) {
			// logo type: text

			// text for logo
			$logo_text = get_theme_mod( 'orin_logo_text', 'ORIN' );
			if ( '' !== $logo_text ) {
				// show text logo
				?>

				<!-- logo (text) -->
				<div class="bwp-logo-text-container">
					<a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home" class="bwp-logo-text"><?php echo esc_html( $logo_text ); ?></a>
				</div>
				<!-- end: logo (text) -->

				<?php
			}

		} else {
			// logo type: image

			if ( function_exists( 'the_custom_logo' ) ) {
				if ( has_custom_logo() ) {
					// logo for light style (default logo)
					$custom_logo_id = get_theme_mod( 'custom_logo' );
					$custom_logo_url = wp_get_attachment_image_url( $custom_logo_id, 'full' );
					// logo for dark style (optional)
					$dark_logo_url = get_theme_mod( 'orin_dark_logo_image' );
					// logo alt
					$custom_logo_alt = get_post_meta( $custom_logo_id, '_wp_attachment_image_alt', true );
					// get current theme style
					$current_theme_style = orin_get_current_theme_style(); // light or dark
					?>

					<!-- logo (image) -->
					<div id="bwp-custom-logo" class="bwp-logo-image-container"
						 data-logo-url="<?php echo esc_url( $custom_logo_url ); ?>"
						 data-dark-logo-url="<?php if ( $dark_logo_url ) { echo esc_url( $dark_logo_url ); } else { echo 'none'; } ?>"
						 data-logo-alt="<?php if ( $custom_logo_alt ) { echo esc_attr( $custom_logo_alt ); } else { echo esc_attr( get_bloginfo( 'name' ) ); } ?>">

						<?php
						if ( 'dark' === $current_theme_style && $dark_logo_url ) {
							// show logo for dark version
							?>

							<!-- logo for dark style -->
							<a href="<?php echo esc_url( home_url( '/' ) ); ?>" class="custom-logo-link" rel="home" aria-current="page">
								<img src="<?php echo esc_url( $dark_logo_url ); ?>" class="custom-logo" alt="<?php if ( $custom_logo_alt ) { echo esc_attr( $custom_logo_alt ); } else { echo esc_attr( get_bloginfo( 'name' ) ); } ?>">
							</a>
							<!-- end: logo for dark style -->

							<?php
						} else {
							// show logo for light version
							the_custom_logo();
						}
						?>

					</div>
					<!-- end: logo (image) -->

					<?php
				} else {
					// if there is no logo, then output an empty block (only for JavaScript needs)
					?>

					<!-- logo (empty block) -->
					<div id="bwp-custom-logo" data-logo-url="none" data-dark-logo-url="none" data-logo-alt="none"></div>
					<!-- end: logo (empty block) -->

					<?php
				}
			}

		} // end: logo type
	}
}


/**
 * The function displays the current page number
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_show_current_page_number' ) ) {
	function orin_show_current_page_number() {
		$show_page_number = get_theme_mod( 'orin_posts_show_page_number', 1 ); // 1 or 0
		if ( $show_page_number ) {
			// get the maximum number of pages
			global $wp_query;
			$total_pages = $wp_query->max_num_pages;
			if ( $total_pages > 1 ) {
				// get the current page number
				$current_page_number = ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;
				?>

				<span class="bwp-page-number">
					<?php echo esc_html__( 'Page №', 'orin' ) . (int) $current_page_number; ?>
				</span>

				<?php
			}
		}
	}
}


/**
 * The function displays a heading for all archive pages
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_show_archive_header' ) ) {
	function orin_show_archive_header() {
		// homepage (section with posts)
		if ( is_home() ) {
			// title and subtitle
			$posts_section_title = get_theme_mod( 'orin_posts_section_title', 'Recent Articles' );
			if ( '' !== $posts_section_title ) {
				$posts_section_subtitle = get_theme_mod( 'orin_posts_section_subtitle', '' );
				?>

				<!-- separator (with gradient) -->
				<div class="bwp-separator bwp-gradient"><span class="bwp-rhomb"></span></div>

				<!-- title and subtitle -->
				<header class="bwp-section-header">
					<h2 class="bwp-section-title">
						<?php
						// title
						echo wp_kses( $posts_section_title, array(
							'a'			=> array(
								'href'		=> array(),
								'title'		=> array(),
								'target'	=> array(),
								'class'		=> array(),
								'rel'		=> array(),
							),
							'span'		=> array(
								'class'		=> array(),
							),
							'strong'	=> array(),
							'b'			=> array(),
							'em'		=> array(),
							'i'			=> array(
								'class'		=> array(),
							),
						) );
						// current page number
						orin_show_current_page_number();
						?>
					</h2>
					<?php
					if ( '' !== $posts_section_subtitle ) {
						?>
						<p>
							<?php
							// subtitle
							echo wp_kses( $posts_section_subtitle, array(
								'a'			=> array(
									'href'		=> array(),
									'title'		=> array(),
									'target'	=> array(),
									'class'		=> array(),
									'rel'		=> array(),
								),
								'span'		=> array(
									'class'		=> array(),
								),
								'strong'	=> array(),
								'b'			=> array(),
								'em'		=> array(),
								'i'			=> array(
									'class'		=> array(),
								),
							) );
							?>
						</p>
						<?php
					}
					?>
					<div class="bwp-section-header-separator"></div>
				</header>
				<!-- end: title and subtitle -->

				<?php
			}
			// end: homepage; start: author page
		} elseif ( is_author() ) {
			// get author name
			$author_display_name = get_the_author_meta( 'display_name' );
			?>

			<!-- separator (with gradient) -->
			<div class="bwp-separator bwp-gradient"><span class="bwp-rhomb"></span></div>

			<!-- title and subtitle (author page) -->
			<header class="bwp-section-header">
				<h2 class="bwp-section-title">
					<?php
					// author name
					echo esc_html( $author_display_name );
					// current page number
					orin_show_current_page_number();
					?>
				</h2>
				<?php if ( get_the_author_meta( 'description' ) ) { ?>
					<div class="bwp-archive-description">
						<?php the_author_meta( 'description' ); ?>
					</div>
				<?php } ?>
				<div class="bwp-section-header-separator"></div>
			</header>
			<!-- end: title and subtitle -->

			<?php
			// end: author page; start: category or tag page
		} elseif ( is_category() || is_tag() ) {
			?>

			<!-- separator (with gradient) -->
			<div class="bwp-separator bwp-gradient"><span class="bwp-rhomb"></span></div>

			<!-- title and subtitle (category/tag page) -->
			<header class="bwp-section-header">
				<h2 class="bwp-category-tag-title bwp-section-title">
					<?php
					// title (category or tag name)
					if ( is_category() ) {
						// show category title
						single_cat_title( '<span class="bwp-category-title-prefix">' . esc_html__( 'Category:', 'orin' ) . '</span>' );
					} elseif ( is_tag() ) {
						// show tag title
						single_tag_title( '<span class="bwp-tag-title-prefix">' . esc_html__( 'Tag:', 'orin' ) . '</span>' );
					}
					// current page number
					orin_show_current_page_number();
					?>
				</h2>
				<?php if ( get_the_archive_description() ) { ?>
					<div class="bwp-archive-description">
						<?php the_archive_description(); ?>
					</div>
				<?php } ?>
				<div class="bwp-section-header-separator"></div>
			</header>
			<!-- end: title and subtitle -->

			<?php
			// end: category or tag page; start: search results page
		} elseif ( is_search() ) {
			// the number of found posts
			global $wp_query;
			$search_results_number_escaped = (int) $wp_query->found_posts; // this variable has been safely escaped
			?>

			<!-- separator (with gradient) -->
			<div class="bwp-separator bwp-gradient"><span class="bwp-rhomb"></span></div>

			<!-- title and subtitle (search results page) -->
			<header class="bwp-section-header bwp-search-results">
				<h2 class="bwp-section-title">
					<?php
					// title
					echo esc_html__( 'Search results for:', 'orin' ) . '<span class="bwp-search-query-text">' . esc_html( get_search_query() ) . '</span>';
					// current page number
					orin_show_current_page_number();
					?>
				</h2>
				<?php if ( 0 !== $search_results_number_escaped ) { ?>
					<div class="bwp-archive-description">
						<?php echo esc_html__( 'Total posts and pages found:', 'orin' ) . '<span class="bwp-search-results-number">' . $search_results_number_escaped . '</span>'; ?>
					</div>
				<?php } ?>
				<div class="bwp-section-header-separator"></div>
			</header>
			<!-- end: title and subtitle -->

			<?php
			// end: search results page; start: archive page
		} elseif ( is_archive() ) {
			?>

			<!-- separator (with gradient) -->
			<div class="bwp-separator bwp-gradient"><span class="bwp-rhomb"></span></div>

			<!-- title and subtitle (archive page) -->
			<header class="bwp-section-header">
				<h2 class="bwp-section-title">
					<?php
					// title
					the_archive_title();
					// current page number
					orin_show_current_page_number();
					?>
				</h2>
				<div class="bwp-section-header-separator"></div>
			</header>
			<!-- end: title and subtitle -->

			<?php
			// end: archive page; start: other pages
		} else {
			?>

			<h2 class="screen-reader-text">
				<?php esc_html_e( 'Blog posts', 'orin' ); ?>
			</h2>

			<?php
		}
	}
}


/**
 * The function displays post metadata (author, date, and categories)
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_show_post_metadata' ) ) {
	function orin_show_post_metadata( $post_type = 'post' ) {
		?>

		<!-- metadata -->
		<ul class="bwp-post-metadata list-unstyled">

			<?php
			// 1: author
			// author data: id, name, and author posts url
			$author_id = get_the_author_meta( 'ID' );
			$author_display_name = get_the_author_meta( 'display_name' );
			$author_posts_url = get_author_posts_url( $author_id );
			?>

			<!-- author -->
			<li class="bwp-author bwp-hidden">
				<a href="<?php echo esc_url( $author_posts_url ); ?>" rel="author">
					<span class="vcard author">
						<span class="fn"><?php echo esc_html( $author_display_name ); ?></span>
					</span>
				</a>
			</li>
			<!-- end: author -->

			<?php
			// 2: date
			// post type: post
			if ( 'post' === $post_type ) {
				// year, month, day
				$time_year = get_the_time( 'Y' );
				$time_month = get_the_time( 'm' );
				$time_day = get_the_time( 'd' );
				?>

				<!-- date (with link) -->
				<li class="bwp-date bwp-hidden">
					<a href="<?php echo esc_url( get_day_link( $time_year, $time_month, $time_day ) ); ?>">
						<span class="date updated"><?php the_time( get_option( 'date_format' ) ); ?></span>
					</a>
				</li>
				<!-- end: date -->

				<?php
			} else {
				// post type: page
				?>

				<!-- date (no link) -->
				<li class="bwp-date bwp-visible">
					<span class="date updated"><?php the_time( get_option( 'date_format' ) ); ?></span>
				</li>
				<!-- end: date -->

				<?php
			}

			// 3: categories
			// show categories for post only
			if ( 'post' === $post_type ) {
				?>

				<!-- categories -->
				<li class="bwp-categories bwp-visible">
					<?php the_category( ', ' ); ?>
				</li>
				<!-- end: categories -->

				<?php
			}
			?>

		</ul>
		<!-- end: metadata -->

		<?php
	}
}


/**
 * The function displays a content of a post (post title and post excerpt)
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_show_post_content' ) ) {
	function orin_show_post_content() {

		// title
		if ( get_the_title() ) {
			?>
			<!-- title -->
			<h3 class="bwp-post-title entry-title">
				<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
			</h3>
			<!-- end: title -->
			<?php
		}
		?>

		<!-- excerpt -->
		<div class="bwp-post-excerpt entry-content">
			<?php the_excerpt(); ?>
		</div>
		<!-- end: excerpt -->

		<?php
	}
}


/**
 * The function displays the "Read more" link and counters at the bottom of each post
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_show_post_links' ) ) {
	function orin_show_post_links( $post_type = 'post', $is_password_protected = false ) {
		// read more: show or hide
		$show_read_more_opt = get_theme_mod( 'orin_posts_show_read_more', 1 ); // 1 or 0
		$read_more_text = get_theme_mod( 'orin_posts_read_more_text', 'Read More' );
		$show_read_more = $show_read_more_opt && '' !== $read_more_text; // true or false
		// views counter: show or hide
		$show_views_counter_opt = get_theme_mod( 'orin_posts_show_views', 1 ); // 1 or 0
		$show_views_counter = function_exists( 'orin_additional_features_show_views_counter' ) && $show_views_counter_opt; // true or false
		// comments counter: show or hide
		$show_comments_counter_opt = get_theme_mod( 'orin_posts_show_comments_counter', 1 ); // 1 or 0
		$show_comments_counter = function_exists( 'orin_additional_features_show_comments_counter' ) && $show_comments_counter_opt; // true or false

		if ( 'post' === $post_type && ( $show_read_more || $show_views_counter || $show_comments_counter ) ) {
			?>

			<!-- read more and counters -->
			<div class="clearfix<?php if ( ! $show_read_more ) { echo ' bwp-post-no-read-more'; } ?>">

				<?php
				// read more
				if ( $show_read_more ) {
					?>
					<!-- read more -->
					<div class="bwp-post-read-more">
						<a href="<?php the_permalink(); ?>"><?php echo esc_html( $read_more_text ); ?></a>
					</div>
					<!-- end: read more -->
					<?php
				}

				// counters: views and comments
				if ( ! $is_password_protected && ( $show_views_counter || $show_comments_counter ) ) {
					?>

					<!-- counters -->
					<div class="bwp-post-counters">

						<?php
						// number of views
						if ( $show_views_counter ) {
							orin_additional_features_show_views_counter( get_the_ID() );
						}

						// number of comments
						if ( $show_comments_counter && ( comments_open() || get_comments_number() ) ) {
							orin_additional_features_show_comments_counter();
						}
						?>

					</div>
					<!-- end: counters -->

					<?php
				}
				?>

			</div>
			<!-- end: read more and counters -->

			<?php
		} elseif ( 'page' === $post_type && $show_read_more ) {
			?>

			<!-- read more -->
			<div class="bwp-post-read-more">
				<a href="<?php the_permalink(); ?>"><?php echo esc_html( $read_more_text ); ?></a>
			</div>
			<!-- end: read more -->

			<?php
		}
	}
}


/**
 * The function displays post metadata on a single page of that post
 *
 * Single post page: author, date (with link), and categories
 * Page: author and date (no link)
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_show_single_post_metadata' ) ) {
	function orin_show_single_post_metadata( $post_type = 'post' ) {
		if ( 'post' === $post_type ) {
			// single post page: show or hide metadata
			$show_author = get_theme_mod( 'orin_single_show_author', 0 ); // 1 or 0
			$show_date = get_theme_mod( 'orin_single_show_date', 1 ); // 1 or 0
			$show_categories = get_theme_mod( 'orin_single_show_categories', 1 ); // 1 or 0
			$metadata_hidden = ( ! $show_author && ! $show_date && ! $show_categories ) ? 'bwp-hidden' : '';
		} else {
			// regular page: show or hide metadata
			$show_author = get_theme_mod( 'orin_page_show_author', 1 ); // 1 or 0
			$show_date = get_theme_mod( 'orin_page_show_date', 1 ); // 1 or 0
			$show_categories = 0;
			$metadata_hidden = ( ! $show_author && ! $show_date ) ? 'bwp-hidden' : '';
		}
		?>

		<!-- metadata -->
		<ul class="bwp-single-post-metadata list-unstyled<?php if ( '' !== $metadata_hidden ) { echo ' ' . sanitize_html_class( $metadata_hidden ); } ?>">

			<?php
			// 1: author
			// author data: id, name, and author posts url
			$author_id = get_the_author_meta( 'ID' );
			$author_display_name = get_the_author_meta( 'display_name' );
			$author_posts_url = get_author_posts_url( $author_id );
			?>

			<!-- author -->
			<li class="bwp-author<?php if ( ! $show_author ) { echo ' bwp-hidden'; } ?>">
				<a href="<?php echo esc_url( $author_posts_url ); ?>" rel="author">
					<span class="vcard author">
						<span class="fn"><?php echo esc_html( $author_display_name ); ?></span>
					</span>
				</a>
			</li>
			<!-- end: author -->

			<?php
			// 2: date
			// post type: post
			if ( 'post' === $post_type ) {
				// year, month, day
				$time_year = get_the_time( 'Y' );
				$time_month = get_the_time( 'm' );
				$time_day = get_the_time( 'd' );
				?>

				<!-- date (with link) -->
				<li class="bwp-date<?php if ( ! $show_date ) { echo ' bwp-hidden'; } ?>">
					<a href="<?php echo esc_url( get_day_link( $time_year, $time_month, $time_day ) ); ?>">
						<span class="date updated"><?php the_time( get_option( 'date_format' ) ); ?></span>
					</a>
				</li>
				<!-- end: date -->

				<?php
			} else {
				// post type: page
				?>

				<!-- date (no link) -->
				<li class="bwp-date<?php if ( ! $show_date ) { echo ' bwp-hidden'; } ?>">
					<span class="date updated"><?php the_time( get_option( 'date_format' ) ); ?></span>
				</li>
				<!-- end: date -->

				<?php
			}

			// 3: categories
			// show categories for post only
			if ( 'post' === $post_type ) {
				?>

				<!-- categories -->
				<li class="bwp-categories<?php if ( ! $show_categories ) { echo ' bwp-hidden'; } ?>">
					<?php the_category( ', ' ); ?>
				</li>
				<!-- end: categories -->

				<?php
			}
			?>

		</ul>
		<!-- end: metadata -->

		<?php
	}
}


/**
 * The function displays tags
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_show_single_post_tags' ) ) {
	function orin_show_single_post_tags() {
		?>

		<!-- tags -->
		<div class="bwp-single-post-tags">
			<?php the_tags( '', '', '' ); ?>
		</div>
		<!-- end: tags -->

		<?php
	}
}


/**
 * The function displays the "About the author" section
 *
 * @since Orin 1.0
 */
if ( ! function_exists( 'orin_show_about_the_author' ) ) {
	function orin_show_about_the_author() {
		// current author data
		$author_id = get_the_author_meta( 'ID' );
		$author_display_name = get_the_author_meta( 'display_name' );
		$author_posts_url = get_author_posts_url( $author_id );
		$author_posts_num_escaped = (int) get_the_author_posts(); // this variable has been safely escaped
		// show or hide avatars (WordPress Settings > Discussion > Avatars > Avatar Display: Show Avatars)
		$show_avatars = get_option( 'show_avatars' );
		?>

		<!-- about the author -->
		<div class="bwp-about-author<?php echo ( ! $show_avatars ) ? ' bwp-no-avatars' : ''; ?>">
			<div class="bwp-separator bwp-gradient"><span class="bwp-rhomb"></span></div>
			<div class="bwp-about-author-container">

				<!-- author data: avatar, name, number of posts -->
				<div class="clearfix">

					<?php if ( $show_avatars ) { ?>
						<figure class="bwp-about-author-avatar">
							<a href="<?php echo esc_url( $author_posts_url ); ?>">
								<?php echo get_avatar( $author_id, '124', '', esc_attr( $author_display_name ) ); ?>
								<div class="bwp-avatar-overlay"></div>
							</a>
						</figure>
					<?php } ?>

					<h3 class="bwp-about-author-name">
						<a href="<?php echo esc_url( $author_posts_url ); ?>"><?php echo esc_html( $author_display_name ); ?></a>
					</h3>

					<?php if ( 0 !== $author_posts_num_escaped ) { ?>
						<div class="bwp-about-author-posts-num">
							<?php echo esc_html__( 'Total posts created:', 'orin' ) . ' ' . $author_posts_num_escaped; ?>
						</div>
					<?php } ?>

				</div>
				<!-- end: author data -->

				<?php if ( get_the_author_meta( 'description' ) ) { ?>
					<!-- biographical information -->
					<div class="bwp-about-author-bio">
						<?php the_author_meta( 'description' ); ?>
					</div>
					<!-- end: biographical information -->
				<?php } ?>

			</div>
		</div>
		<!-- end: about the author -->

		<?php
	}
}


/**
 * Customizer (Theme options)
 *
 * @since Orin 1.0
 */
require_once get_theme_file_path( '/assets/customizer.php' );
require_once get_theme_file_path( '/assets/sanitize-functions.php' );
require_once get_theme_file_path( '/assets/inline-styles.php' );
