<?php
/**
 * Data sanitization functions
 *
 * @package WordPress
 * @subpackage Orin
 * @since Orin 1.0
 */

/**
 * The function checks all numeric values
 *
 * @since Orin 1.0
 */
function orin_sanitize_number_intval( $input ) {
	if ( is_numeric( $input ) && $input >= 1 ) {
		return intval( $input );
	} else {
		return '';
	}
}


/**
 * The function checks the checkboxes
 *
 * @since Orin 1.0
 */
function orin_sanitize_checkbox( $input ) {
	if ( 1 === (int) $input ) {
		return 1;
	} else {
		return 0;
	}
}


/**
 * The function checks the type of logo
 *
 * @since Orin 1.0
 */
function orin_sanitize_logo_type( $input ) {
	$valid = array(
		'text'	=> 'Text',
		'image'	=> 'Image',
	);

	if ( array_key_exists( $input, $valid ) ) {
		return $input;
	} else {
		return 'text'; // default
	}
}


/**
 * The function checks the type of posts for the homepage slider
 *
 * @since Orin 1.0
 */
function orin_sanitize_slider_posts_type( $input ) {
	$valid = array(
		'posts-by-category'	=> 'Posts by category',
		'featured-posts'	=> 'Featured posts',
	);

	if ( array_key_exists( $input, $valid ) ) {
		return $input;
	} else {
		return 'posts-by-category'; // default
	}
}


/**
 * The function checks the categories of posts
 *
 * @since Orin 1.0
 */
function orin_sanitize_categories( $input ) {
	$categories = array();
	$categories_obj = get_categories( 'hide_empty=0&depth=1&type=post' );
	foreach ( $categories_obj as $category ) {
		$category_id = $category->term_id;
		$category_name = $category->cat_name;
		$categories[ intval( $category_id ) ] = esc_html( $category_name );
	}
	$categories = array( 0 => 'All categories' ) + $categories;

	if ( array_key_exists( $input, $categories ) ) {
		return $input;
	} else {
		return 0; // All categories
	}
}


/**
 * The function checks the order values for the homepage slider items
 *
 * @since Orin 1.0
 */
function orin_sanitize_slider_orderby( $input ) {
	$valid = array(
		'rand'			=> 'Random order',
		'date'			=> 'Publication date',
		'comment_count'	=> 'Number of comments',
	);

	if ( array_key_exists( $input, $valid ) ) {
		return $input;
	} else {
		return 'date'; // default
	}
}


/**
 * The function checks the transition type for the slider items
 *
 * @since Orin 1.0
 */
function orin_sanitize_slider_transition_type( $input ) {
	$valid = array(
		'fade'				=> 'Fade animation',
		'slide_horizontal'	=> 'Slide horizontal',
		'slide_vertical'	=> 'Slide vertical',
	);

	if ( array_key_exists( $input, $valid ) ) {
		return $input;
	} else {
		return 'slide_vertical'; // default
	}
}


/**
 * The function checks pages with posts where the first post needs to be enlarged
 *
 * @since Orin 1.0
 */
function orin_sanitize_pages_with_posts( $input ) {
	$valid = array(
		'homepage'		=> 'Only on the homepage',
		'all-archives'	=> 'On all pages with posts',
		'no-increase'	=> 'Do not increase',
	);

	if ( array_key_exists( $input, $valid ) ) {
		return $input;
	} else {
		return 'homepage'; // default
	}
}


/**
 * The function checks the link type on the featured image
 *
 * @since Orin 1.0
 */
function orin_sanitize_featured_image_link_type( $input ) {
	$valid = array(
		'link_to_post'	=> 'Link to post',
		'link_to_image'	=> 'Link to image',
	);

	if ( array_key_exists( $input, $valid ) ) {
		return $input;
	} else {
		return 'link_to_post'; // default
	}
}


/**
 * The function checks the type of fonts
 *
 * @since Orin 1.1
 */
function orin_sanitize_font_type( $input ) {
	$valid = array(
		'sans-serif'	=> 'Sans-serif fonts',
		'serif'			=> 'Serif fonts',
	);

	if ( array_key_exists( $input, $valid ) ) {
		return $input;
	} else {
		return 'sans-serif'; // default
	}
}


/**
 * The function checks the theme's color scheme
 *
 * @since Orin 1.1
 */
function orin_sanitize_color_scheme( $input ) {
	$valid = array(
		'light'	=> 'Light',
		'dark'	=> 'Dark',
	);

	if ( array_key_exists( $input, $valid ) ) {
		return $input;
	} else {
		return 'light'; // default
	}
}


/**
 * The function checks the type of cookies notification on mobile devices
 *
 * @since Orin 1.0
 */
function orin_sanitize_cookies_notice_on_mobile( $input ) {
	$valid = array(
		'hidden'	=> 'Hidden notification',
		'visible'	=> 'Visible notification',
	);

	if ( array_key_exists( $input, $valid ) ) {
		return $input;
	} else {
		return 'hidden'; // default
	}
}


/**
 * Filtering valid HTML tags (p: class; a: href, title, target, class, rel; span: class; strong; b; em; i: class; br)
 *
 * @since Orin 1.0
 */
function orin_sanitize_wp_kses_text( $input ) {
	return wp_kses( $input, array(
		'p'			=> array(
			'class'		=> array(),
		),
		'a'			=> array(
			'href'		=> array(),
			'title'		=> array(),
			'target'	=> array(),
			'class'		=> array(),
			'rel'		=> array(),
		),
		'span'		=> array(
			'class'		=> array(),
		),
		'strong'	=> array(),
		'b'			=> array(),
		'em'		=> array(),
		'i'			=> array(
			'class'		=> array(),
		),
		'br'		=> array(),
	) );
}
