<?php
/**
 * Theme options (Appearance > Customize)
 *
 * Additional sanitize functions are located in the "sanitize-functions.php" file (/orin/assets/sanitize-functions.php)
 *
 * @package WordPress
 * @subpackage Orin
 * @since Orin 1.0
 */

function orin_customize_register( $wp_customize ) {

	/**
	 * Customize control: Heading (title and subtitle)
	 * -------------------------------------------------------------
	 */
	class orin_customize_control_heading extends WP_Customize_Control {
		public $type = 'orin_heading';
		public function render_content() {
			?>

			<div class="bwp-customize-control-heading">
				<span class="customize-control-title">
					<?php echo esc_html( $this->label ); ?>
				</span>
				<?php if ( $this->value() ) { ?>
					<span class="bwp-customize-control-subtitle">
						<?php echo esc_html( $this->value() ); ?>
					</span>
				<?php } ?>
			</div>

			<?php
		}
	}


	/**
	 * Customize control: Number field (input type = number; min=1, max=10000)
	 * -------------------------------------------------------------
	 */
	class orin_customize_control_number extends WP_Customize_Control {
		public $type = 'orin_number_field';
		public function render_content() {
			?>

			<label>
				<span class="customize-control-title">
					<?php echo esc_html( $this->label ); ?>
				</span>
				<input type="number" min="1" max="10000" value="<?php echo (int) $this->value(); ?>" <?php $this->link(); ?>>
			</label>

			<?php
		}
	}


	/**
	 * Customize control: Short description
	 * -------------------------------------------------------------
	 */
	class orin_customize_control_description extends WP_Customize_Control {
		public $type = 'orin_description';
		public function render_content() {
			?>

			<div class="bwp-customize-control-description">
				<?php
				echo esc_html( $this->value() );
				?>
			</div>

			<?php
		}
	}


	/**
	 * Array with categories
	 * -------------------------------------------------------------
	 */
	$categories = array();
	$categories_obj = get_categories( 'hide_empty=0&depth=1&type=post' );
	foreach ( $categories_obj as $category ) {
		$category_id = $category->term_id;
		$category_name = $category->cat_name;
		$categories[ intval( $category_id ) ] = esc_html( $category_name );
	}
	$categories = array( 0 => esc_html__( 'All categories', 'orin' ) ) + $categories;


	/**
	 * Theme Options
	 * -------------------------------------------------------------
	 *
	 * 1.0 - Site Identity
	 * -------------------------------------------------------------
	 */

	// Logo for dark style
	$wp_customize->add_setting(
		'orin_dark_logo_image',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_url',
		)
	);

	$wp_customize->add_control(
		new WP_Customize_Image_Control(
			$wp_customize,
			'orin_dark_logo_image',
			array(
				'label'		=> __( 'Logo For Dark Style (optional)', 'orin' ),
				'section'	=> 'title_tagline',
				'settings'	=> 'orin_dark_logo_image',
			)
		)
	);


	// Logo for dark style: Description
	$wp_customize->add_setting(
		'orin_dark_logo_image_desc',
		array(
			'default'			=> __( 'The logo for the dark style should have the same dimensions as the original logo and should differ only in color.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_dark_logo_image_desc',
			array(
				'label'		=> __( 'Logo For Dark Style (Setting Description)', 'orin' ),
				'section'	=> 'title_tagline',
				'settings'	=> 'orin_dark_logo_image_desc',
			)
		)
	);


	// Logo width
	$wp_customize->add_setting(
		'orin_logo_width',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'orin_sanitize_number_intval',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_number(
			$wp_customize,
			'orin_logo_width',
			array(
				'label'		=> __( 'Logo Width, px', 'orin' ),
				'section'	=> 'title_tagline',
				'settings'	=> 'orin_logo_width',
			)
		)
	);


	/**
	 * 2.0 - Header And Text Logo
	 * -------------------------------------------------------------
	 */

	// Add new section
	$wp_customize->add_section(
		'orin_header_section',
		array(
			'title'		=> __( 'Header And Text Logo', 'orin' ),
			'priority'	=> 21,
		)
	);


	/**
	 * 2.1 - Header And Text Logo: Header
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_header_section_header',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_header_section_header',
			array(
				'label'		=> __( 'Header', 'orin' ),
				'section'	=> 'orin_header_section',
				'settings'	=> 'orin_header_section_header',
			)
		)
	);


	// Sticky header
	$wp_customize->add_setting(
		'orin_enable_sticky_header',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_enable_sticky_header',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Enable sticky header', 'orin' ),
			'section'	=> 'orin_header_section',
		)
	);


	// Enable/disable switching between font types
	$wp_customize->add_setting(
		'orin_enable_font_switch',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_enable_font_switch',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Enable switching between font types (sans-serif or serif fonts)', 'orin' ),
			'section'	=> 'orin_header_section',
		)
	);


	// Enable/disable switching between font types: Description
	$wp_customize->add_setting(
		'orin_font_switch_desc',
		array(
			'default'			=> __( 'You can change the default fonts in the "Styles And Cookies" section: "Styles And Cookies > Fonts And Color Mode > Default Fonts: Sans-Serif Or Serif".', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_font_switch_desc',
			array(
				'label'		=> __( 'Switching Between Font Types (Setting Description)', 'orin' ),
				'section'	=> 'orin_header_section',
				'settings'	=> 'orin_font_switch_desc',
			)
		)
	);


	// Enable/disable color switch
	$wp_customize->add_setting(
		'orin_enable_color_switch',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_enable_color_switch',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Enable color switch (moon/sun icon)', 'orin' ),
			'section'	=> 'orin_header_section',
		)
	);


	// Enable/disable color switch: Description
	$wp_customize->add_setting(
		'orin_color_switch_desc',
		array(
			'default'			=> __( 'You can change the default site style in the "Styles And Cookies" section: "Styles And Cookies > Fonts And Color Mode > Default Site Style: Light Or Dark".', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_color_switch_desc',
			array(
				'label'		=> __( 'Color Switch (Setting Description)', 'orin' ),
				'section'	=> 'orin_header_section',
				'settings'	=> 'orin_color_switch_desc',
			)
		)
	);


	// Show/hide search
	$wp_customize->add_setting(
		'orin_show_search',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_show_search',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show search icon and dropdown search form', 'orin' ),
			'section'	=> 'orin_header_section',
		)
	);


	// Show/hide sidebar button
	$wp_customize->add_setting(
		'orin_show_sidebar_button',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_show_sidebar_button',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show sidebar button (arrow icon)', 'orin' ),
			'section'	=> 'orin_header_section',
		)
	);


	// Show/hide sidebar button: Description
	$wp_customize->add_setting(
		'orin_show_sidebar_button_desc',
		array(
			'default'			=> __( 'The sidebar button will be hidden if there are no widgets in your sidebar.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_show_sidebar_button_desc',
			array(
				'label'		=> __( 'Show sidebar button (Setting description)', 'orin' ),
				'section'	=> 'orin_header_section',
				'settings'	=> 'orin_show_sidebar_button_desc',
			)
		)
	);


	/**
	 * 2.2 - Header And Text Logo: Logo
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_header_section_logo',
		array(
			'default'			=> __( 'By default, the Orin theme displays a text logo. Below you can change the text for your logo or change the logo type from text to image.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_header_section_logo',
			array(
				'label'		=> __( 'Logo', 'orin' ),
				'section'	=> 'orin_header_section',
				'settings'	=> 'orin_header_section_logo',
			)
		)
	);


	// Logo type
	$wp_customize->add_setting(
		'orin_logo_type',
		array(
			'default'			=> 'text',
			'sanitize_callback'	=> 'orin_sanitize_logo_type',
		)
	);

	$wp_customize->add_control(
		'orin_logo_type',
		array(
			'type'		=> 'select',
			'label'		=> __( 'Logo Type', 'orin' ),
			'section'	=> 'orin_header_section',
			'choices'	=> array(
				'text'		=> esc_html__( 'Text', 'orin' ),
				'image'		=> esc_html__( 'Image', 'orin' ),
			),
		)
	);


	// Logo type: Description
	$wp_customize->add_setting(
		'orin_logo_type_desc',
		array(
			'default'			=> __( 'You can add an image for your logo in the "Site Identity" section: "Site Identity > Logo".', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_logo_type_desc',
			array(
				'label'		=> __( 'Logo Type (Setting Description)', 'orin' ),
				'section'	=> 'orin_header_section',
				'settings'	=> 'orin_logo_type_desc',
			)
		)
	);


	// Logo text
	$wp_customize->add_setting(
		'orin_logo_text',
		array(
			'default'			=> 'ORIN',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		'orin_logo_text',
		array(
			'label'		=> __( 'Logo Text (Logo Type: Text)', 'orin' ),
			'section'	=> 'orin_header_section',
			'type'		=> 'text',
		)
	);


	/**
	 * 3.0 - Homepage Slider
	 * -------------------------------------------------------------
	 */

	// Add new section
	$wp_customize->add_section(
		'orin_homepage_slider_section',
		array(
			'title'		=> __( 'Homepage Slider', 'orin' ),
			'priority'	=> 22,
		)
	);


	/**
	 * 3.1 - Homepage Slider: Slider Display Settings
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_slider_display_settings',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_slider_display_settings',
			array(
				'label'		=> __( 'Slider Display Settings', 'orin' ),
				'section'	=> 'orin_homepage_slider_section',
				'settings'	=> 'orin_slider_display_settings',
			)
		)
	);


	// Show/hide slider
	$wp_customize->add_setting(
		'orin_show_homepage_slider',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_show_homepage_slider',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show homepage slider', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
		)
	);


	// Show the slider only on the first pagination page
	$wp_customize->add_setting(
		'orin_show_slider_on_first_page',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_show_slider_on_first_page',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show the slider only on the first pagination page', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
		)
	);


	/**
	 * 3.2 - Homepage Slider: Title And Subtitle
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_slider_title_subtitle_settings',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_slider_title_subtitle_settings',
			array(
				'label'		=> __( 'Slider Section: Title And Subtitle', 'orin' ),
				'section'	=> 'orin_homepage_slider_section',
				'settings'	=> 'orin_slider_title_subtitle_settings',
			)
		)
	);


	// Slider Section: Title
	$wp_customize->add_setting(
		'orin_slider_section_title',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'orin_sanitize_wp_kses_text',
		)
	);

	$wp_customize->add_control(
		'orin_slider_section_title',
		array(
			'label'		=> __( 'Slider Section: Title', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
			'type'		=> 'text',
		)
	);


	// Allowed HTML tags (Slider Section: Title)
	$wp_customize->add_setting(
		'orin_slider_section_title_desc',
		array(
			'default'			=> __( 'Allowed HTML tags: a, span, strong, b, em, i. Allowed attributes: a - href, title, target, class, rel. span - class. i - class.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_slider_section_title_desc',
			array(
				'label'		=> __( 'Slider Section: Title - Allowed HTML Tags', 'orin' ),
				'section'	=> 'orin_homepage_slider_section',
				'settings'	=> 'orin_slider_section_title_desc',
			)
		)
	);


	// Slider Section: Subtitle
	$wp_customize->add_setting(
		'orin_slider_section_subtitle',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'orin_sanitize_wp_kses_text',
		)
	);

	$wp_customize->add_control(
		'orin_slider_section_subtitle',
		array(
			'label'		=> __( 'Slider Section: Subtitle', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
			'type'		=> 'textarea',
		)
	);


	// Allowed HTML tags (Slider Section: Subtitle)
	$wp_customize->add_setting(
		'orin_slider_section_subtitle_desc',
		array(
			'default'			=> __( 'Allowed HTML tags: a, span, strong, b, em, i. Allowed attributes: a - href, title, target, class, rel. span - class. i - class.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_slider_section_subtitle_desc',
			array(
				'label'		=> __( 'Slider Section: Subtitle - Allowed HTML Tags', 'orin' ),
				'section'	=> 'orin_homepage_slider_section',
				'settings'	=> 'orin_slider_section_subtitle_desc',
			)
		)
	);


	/**
	 * 3.3 - Homepage Slider: Slider Options
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_slider_options',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_slider_options',
			array(
				'label'		=> __( 'Slider Options', 'orin' ),
				'section'	=> 'orin_homepage_slider_section',
				'settings'	=> 'orin_slider_options',
			)
		)
	);


	// Maximum number of slides
	$wp_customize->add_setting(
		'orin_slider_slides_max_number',
		array(
			'default'			=> 4,
			'sanitize_callback'	=> 'orin_sanitize_number_intval',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_number(
			$wp_customize,
			'orin_slider_slides_max_number',
			array(
				'label'		=> __( 'Maximum Number Of Slides (Default Value: 4)', 'orin' ),
				'section'	=> 'orin_homepage_slider_section',
				'settings'	=> 'orin_slider_slides_max_number',
			)
		)
	);


	// Type of posts for homepage slider
	$wp_customize->add_setting(
		'orin_slider_shows_posts_from',
		array(
			'default'			=> 'posts-by-category',
			'sanitize_callback'	=> 'orin_sanitize_slider_posts_type',
		)
	);

	$wp_customize->add_control(
		'orin_slider_shows_posts_from',
		array(
			'type'		=> 'select',
			'label'		=> __( 'What To Show In The Slider?', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
			'choices'	=> array(
				'posts-by-category'	=> esc_html__( 'Posts by category', 'orin' ),
				'featured-posts'	=> esc_html__( 'Featured posts', 'orin' ),
			),
		)
	);


	// Type of posts for homepage slider: Description
	$wp_customize->add_setting(
		'orin_slider_shows_posts_from_desc',
		array(
			'default'			=> __( '"Posts by category" - The slider will display posts from all categories or from one selected category. "Featured posts" - The slider will only display featured posts. You can select these posts manually in the settings for each post: "WordPress Dashboard > Posts > Add new or Edit existing post > Additional Settings > Homepage slider".', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_slider_shows_posts_from_desc',
			array(
				'label'		=> __( 'What To Show In The Slider? (Setting Description)', 'orin' ),
				'section'	=> 'orin_homepage_slider_section',
				'settings'	=> 'orin_slider_shows_posts_from_desc',
			)
		)
	);


	// Category for slider items
	$wp_customize->add_setting(
		'orin_slider_category_id',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_categories',
		)
	);

	$wp_customize->add_control(
		'orin_slider_category_id',
		array(
			'type'		=> 'select',
			'label'		=> __( 'Category For Slider Items', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
			'choices'	=> $categories,
		)
	);


	// Category for slider items: Description
	$wp_customize->add_setting(
		'orin_slider_category_id_desc',
		array(
			'default'			=> __( 'The category selection setting works only if in the slider you display posts by category.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_slider_category_id_desc',
			array(
				'label'		=> __( 'Category For Slider Items (Setting Description)', 'orin' ),
				'section'	=> 'orin_homepage_slider_section',
				'settings'	=> 'orin_slider_category_id_desc',
			)
		)
	);


	// Order by...
	$wp_customize->add_setting(
		'orin_slider_orderby',
		array(
			'default'			=> 'date',
			'sanitize_callback'	=> 'orin_sanitize_slider_orderby',
		)
	);

	$wp_customize->add_control(
		'orin_slider_orderby',
		array(
			'type'		=> 'select',
			'label'		=> __( 'Order By', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
			'choices'	=> array(
				'rand'			=> esc_html__( 'Random order', 'orin' ),
				'date'			=> esc_html__( 'Publication date', 'orin' ),
				'comment_count'	=> esc_html__( 'Number of comments', 'orin' ),
			),
		)
	);


	// Enable popup images for the slider
	$wp_customize->add_setting(
		'orin_slider_popup_image',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_slider_popup_image',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Enable popup images for the slider', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
		)
	);


	// Enable rewind
	$wp_customize->add_setting(
		'orin_slider_enable_rewind',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_slider_enable_rewind',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Enable rewind (Moves to the opposite edge when reaching the first or last slide)', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
		)
	);


	// Transition type for slides
	$wp_customize->add_setting(
		'orin_slider_transition_type',
		array(
			'default'			=> 'slide_vertical',
			'sanitize_callback'	=> 'orin_sanitize_slider_transition_type',
		)
	);

	$wp_customize->add_control(
		'orin_slider_transition_type',
		array(
			'type'		=> 'select',
			'label'		=> __( 'Transition Type For Slides', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
			'choices'	=> array(
				'fade'				=> esc_html__( 'Fade animation', 'orin' ),
				'slide_horizontal'	=> esc_html__( 'Slide horizontal', 'orin' ),
				'slide_vertical'	=> esc_html__( 'Slide vertical', 'orin' ),
			),
		)
	);


	// Speed of the slide animation
	$wp_customize->add_setting(
		'orin_slider_animation_speed',
		array(
			'default'			=> 600,
			'sanitize_callback'	=> 'orin_sanitize_number_intval',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_number(
			$wp_customize,
			'orin_slider_animation_speed',
			array(
				'label'		=> __( 'Speed Of The Slide Animation (in "ms")', 'orin' ),
				'section'	=> 'orin_homepage_slider_section',
				'settings'	=> 'orin_slider_animation_speed',
			)
		)
	);


	// Speed of the slide animation: Description
	$wp_customize->add_setting(
		'orin_slider_animation_speed_desc',
		array(
			'default'			=> __( 'Default value: 600. Recommended range of values: 400...800.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_slider_animation_speed_desc',
			array(
				'label'		=> __( 'Speed Of The Slide Animation (Setting Description)', 'orin' ),
				'section'	=> 'orin_homepage_slider_section',
				'settings'	=> 'orin_slider_animation_speed_desc',
			)
		)
	);


	// Show navigation buttons (Next/Previous)
	$wp_customize->add_setting(
		'orin_slider_show_controls',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_slider_show_controls',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show navigation buttons (Next/Previous)', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
		)
	);


	// Show pagination (Dots)
	$wp_customize->add_setting(
		'orin_slider_show_nav',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_slider_show_nav',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show pagination (Dots)', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
		)
	);


	/**
	 * 3.4 - Homepage Slider: Slider Content
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_slider_content',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_slider_content',
			array(
				'label'		=> __( 'Slider Content', 'orin' ),
				'section'	=> 'orin_homepage_slider_section',
				'settings'	=> 'orin_slider_content',
			)
		)
	);


	// Show author
	$wp_customize->add_setting(
		'orin_slider_show_author',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_slider_show_author',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show author', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
		)
	);


	// Show date
	$wp_customize->add_setting(
		'orin_slider_show_date',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_slider_show_date',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show date', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
		)
	);


	// Show categories
	$wp_customize->add_setting(
		'orin_slider_show_categories',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_slider_show_categories',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show categories', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
		)
	);


	// Show the "Read more" link
	$wp_customize->add_setting(
		'orin_slider_show_read_more',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_slider_show_read_more',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show the "Read more" link', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
		)
	);


	// Text for the "read more" link
	$wp_customize->add_setting(
		'orin_slider_read_more_text',
		array(
			'default'			=> 'Read More',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		'orin_slider_read_more_text',
		array(
			'label'		=> __( 'Text For The "Read More" Link', 'orin' ),
			'section'	=> 'orin_homepage_slider_section',
			'type'		=> 'text',
		)
	);


	/**
	 * 4.0 - Blog Posts
	 * -------------------------------------------------------------
	 */

	// Add new section
	$wp_customize->add_section(
		'orin_blog_posts_section',
		array(
			'title'		=> __( 'Blog Posts', 'orin' ),
			'priority'	=> 23,
		)
	);


	/**
	 * 4.1 - Blog Posts: Title And Subtitle
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_posts_title_subtitle_settings',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_posts_title_subtitle_settings',
			array(
				'label'		=> __( 'Posts Section: Title And Subtitle', 'orin' ),
				'section'	=> 'orin_blog_posts_section',
				'settings'	=> 'orin_posts_title_subtitle_settings',
			)
		)
	);


	// Posts Section: Title
	$wp_customize->add_setting(
		'orin_posts_section_title',
		array(
			'default'			=> 'Recent Articles',
			'sanitize_callback'	=> 'orin_sanitize_wp_kses_text',
		)
	);

	$wp_customize->add_control(
		'orin_posts_section_title',
		array(
			'label'		=> __( 'Posts Section: Title', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
			'type'		=> 'text',
		)
	);


	// Allowed HTML tags (Posts Section: Title)
	$wp_customize->add_setting(
		'orin_posts_section_title_desc',
		array(
			'default'			=> __( 'This title appears on the homepage with your latest posts. Allowed HTML tags: a, span, strong, b, em, i. Allowed attributes: a - href, title, target, class, rel. span - class. i - class.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_posts_section_title_desc',
			array(
				'label'		=> __( 'Posts Section: Title - Allowed HTML Tags', 'orin' ),
				'section'	=> 'orin_blog_posts_section',
				'settings'	=> 'orin_posts_section_title_desc',
			)
		)
	);


	// Posts Section: Subtitle
	$wp_customize->add_setting(
		'orin_posts_section_subtitle',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'orin_sanitize_wp_kses_text',
		)
	);

	$wp_customize->add_control(
		'orin_posts_section_subtitle',
		array(
			'label'		=> __( 'Posts Section: Subtitle', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
			'type'		=> 'textarea',
		)
	);


	// Allowed HTML tags (Posts Section: Subtitle)
	$wp_customize->add_setting(
		'orin_posts_section_subtitle_desc',
		array(
			'default'			=> __( 'This subtitle appears on the homepage with your latest posts. Allowed HTML tags: a, span, strong, b, em, i. Allowed attributes: a - href, title, target, class, rel. span - class. i - class.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_posts_section_subtitle_desc',
			array(
				'label'		=> __( 'Posts Section: Subtitle - Allowed HTML Tags', 'orin' ),
				'section'	=> 'orin_blog_posts_section',
				'settings'	=> 'orin_posts_section_subtitle_desc',
			)
		)
	);


	// Show page number
	$wp_customize->add_setting(
		'orin_posts_show_page_number',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_posts_show_page_number',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show page number', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
		)
	);


	/**
	 * 4.2 - Blog Posts: Posts
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_posts_settings',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_posts_settings',
			array(
				'label'		=> __( 'Posts', 'orin' ),
				'section'	=> 'orin_blog_posts_section',
				'settings'	=> 'orin_posts_settings',
			)
		)
	);


	// Increase the first post in the list of posts
	$wp_customize->add_setting(
		'orin_increase_first_post_on_page',
		array(
			'default'			=> 'homepage',
			'sanitize_callback'	=> 'orin_sanitize_pages_with_posts',
		)
	);

	$wp_customize->add_control(
		'orin_increase_first_post_on_page',
		array(
			'type'		=> 'select',
			'label'		=> __( 'Increase The First Post In The List Of Posts', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
			'choices'	=> array(
				'homepage'		=> esc_html__( 'Only on the homepage', 'orin' ),
				'all-archives'	=> esc_html__( 'On all pages with posts', 'orin' ),
				'no-increase'	=> esc_html__( 'Do not increase', 'orin' ),
			),
		)
	);


	// Increase the first post in the list of posts: Description
	$wp_customize->add_setting(
		'orin_increase_first_post_on_page_desc',
		array(
			'default'			=> __( 'The first post will be increased only on the first pagination page. However, this can be changed in the setting below.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_increase_first_post_on_page_desc',
			array(
				'label'		=> __( 'Increase The First Post In The List Of Posts (Setting Description)', 'orin' ),
				'section'	=> 'orin_blog_posts_section',
				'settings'	=> 'orin_increase_first_post_on_page_desc',
			)
		)
	);


	// Increase the first post on pagination pages or not?
	$wp_customize->add_setting(
		'orin_increase_first_post_on_pagination',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_increase_first_post_on_pagination',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Increase the first post on pagination pages or not?', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
		)
	);


	// Increase the size of small images
	$wp_customize->add_setting(
		'orin_increase_images',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_increase_images',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Increase the size of small images (setting applies only to your featured images)', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
		)
	);


	// Featured image: Link type
	$wp_customize->add_setting(
		'orin_featured_image_link_type',
		array(
			'default'			=> 'link_to_post',
			'sanitize_callback'	=> 'orin_sanitize_featured_image_link_type',
		)
	);

	$wp_customize->add_control(
		'orin_featured_image_link_type',
		array(
			'type'		=> 'select',
			'label'		=> __( 'Featured Image: Link Type', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
			'choices'	=> array(
				'link_to_post'	=> esc_html__( 'Link to post', 'orin' ),
				'link_to_image'	=> esc_html__( 'Link to image', 'orin' ),
			),
		)
	);


	// Enable hover effects when you hover on featured images
	$wp_customize->add_setting(
		'orin_images_hover_effects',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_images_hover_effects',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Enable hover effects when you hover on featured images', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
		)
	);


	// Excerpt length (Number of words)
	$wp_customize->add_setting(
		'orin_excerpt_length',
		array(
			'default'			=> 25,
			'sanitize_callback'	=> 'orin_sanitize_number_intval',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_number(
			$wp_customize,
			'orin_excerpt_length',
			array(
				'label'		=> __( 'Excerpt Length (Number Of Words)', 'orin' ),
				'section'	=> 'orin_blog_posts_section',
				'settings'	=> 'orin_excerpt_length',
			)
		)
	);


	// Show categories
	$wp_customize->add_setting(
		'orin_posts_show_categories',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_posts_show_categories',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show categories', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
		)
	);


	// Show the "Read more" link
	$wp_customize->add_setting(
		'orin_posts_show_read_more',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_posts_show_read_more',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show the "Read more" link', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
		)
	);


	// Text for the "read more" link
	$wp_customize->add_setting(
		'orin_posts_read_more_text',
		array(
			'default'			=> 'Read More',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		'orin_posts_read_more_text',
		array(
			'label'		=> __( 'Text For The "Read More" Link', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
			'type'		=> 'text',
		)
	);


	// if "Orin Additional Features" plugin is activated
	if ( function_exists( 'orin_additional_features_show_views_counter' ) && function_exists( 'orin_additional_features_show_comments_counter' ) ) {

		// Show views counter
		$wp_customize->add_setting(
			'orin_posts_show_views',
			array(
				'default'			=> 1,
				'sanitize_callback'	=> 'orin_sanitize_checkbox',
			)
		);

		$wp_customize->add_control(
			'orin_posts_show_views',
			array(
				'type'		=> 'checkbox',
				'label'		=> __( 'Show views counter', 'orin' ),
				'section'	=> 'orin_blog_posts_section',
			)
		);


		// Show comments counter
		$wp_customize->add_setting(
			'orin_posts_show_comments_counter',
			array(
				'default'			=> 1,
				'sanitize_callback'	=> 'orin_sanitize_checkbox',
			)
		);

		$wp_customize->add_control(
			'orin_posts_show_comments_counter',
			array(
				'type'		=> 'checkbox',
				'label'		=> __( 'Show comments counter', 'orin' ),
				'section'	=> 'orin_blog_posts_section',
			)
		);

	}


	/**
	 * 4.3 - Blog Posts: Gallery Format: Slider Options
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_gallery_slider_options',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_gallery_slider_options',
			array(
				'label'		=> __( 'Gallery Format: Slider Options', 'orin' ),
				'section'	=> 'orin_blog_posts_section',
				'settings'	=> 'orin_gallery_slider_options',
			)
		)
	);


	// Enable rewind
	$wp_customize->add_setting(
		'orin_gallery_slider_enable_rewind',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_gallery_slider_enable_rewind',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Enable rewind (Moves to the opposite edge when reaching the first or last slide)', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
		)
	);


	// Transition type for slides
	$wp_customize->add_setting(
		'orin_gallery_slider_transition_type',
		array(
			'default'			=> 'slide_vertical',
			'sanitize_callback'	=> 'orin_sanitize_slider_transition_type',
		)
	);

	$wp_customize->add_control(
		'orin_gallery_slider_transition_type',
		array(
			'type'		=> 'select',
			'label'		=> __( 'Transition Type For Slides', 'orin' ),
			'section'	=> 'orin_blog_posts_section',
			'choices'	=> array(
				'fade'				=> esc_html__( 'Fade animation', 'orin' ),
				'slide_horizontal'	=> esc_html__( 'Slide horizontal', 'orin' ),
				'slide_vertical'	=> esc_html__( 'Slide vertical', 'orin' ),
			),
		)
	);


	// Speed of the slide animation
	$wp_customize->add_setting(
		'orin_gallery_slider_animation_speed',
		array(
			'default'			=> 600,
			'sanitize_callback'	=> 'orin_sanitize_number_intval',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_number(
			$wp_customize,
			'orin_gallery_slider_animation_speed',
			array(
				'label'		=> __( 'Speed Of The Slide Animation (in "ms")', 'orin' ),
				'section'	=> 'orin_blog_posts_section',
				'settings'	=> 'orin_gallery_slider_animation_speed',
			)
		)
	);


	// Speed of the slide animation: Description
	$wp_customize->add_setting(
		'orin_gallery_slider_animation_speed_desc',
		array(
			'default'			=> __( 'Default value: 600. Recommended range of values: 400...800.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_gallery_slider_animation_speed_desc',
			array(
				'label'		=> __( 'Speed Of The Slide Animation (Setting Description)', 'orin' ),
				'section'	=> 'orin_blog_posts_section',
				'settings'	=> 'orin_gallery_slider_animation_speed_desc',
			)
		)
	);


	/**
	 * 5.0 - Single Pages
	 * -------------------------------------------------------------
	 */

	// Add new section
	$wp_customize->add_section(
		'orin_single_pages_section',
		array(
			'title'		=> __( 'Single Pages', 'orin' ),
			'priority'	=> 24,
		)
	);


	/**
	 * 5.1 - Single Pages: Single Post Page
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_single_post_settings',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_single_post_settings',
			array(
				'label'		=> __( 'Single Post Page', 'orin' ),
				'section'	=> 'orin_single_pages_section',
				'settings'	=> 'orin_single_post_settings',
			)
		)
	);


	// Show author
	$wp_customize->add_setting(
		'orin_single_show_author',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_single_show_author',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show author', 'orin' ),
			'section'	=> 'orin_single_pages_section',
		)
	);


	// Show date
	$wp_customize->add_setting(
		'orin_single_show_date',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_single_show_date',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show date', 'orin' ),
			'section'	=> 'orin_single_pages_section',
		)
	);


	// Show categories
	$wp_customize->add_setting(
		'orin_single_show_categories',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_single_show_categories',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show categories', 'orin' ),
			'section'	=> 'orin_single_pages_section',
		)
	);


	// Featured image (or Slider images): Show cropped image instead of full image
	$wp_customize->add_setting(
		'orin_single_cropped_image',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_single_cropped_image',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Featured image (or Slider images): Show cropped image instead of full image (Note: This setting applies to a single post page as well as a regular page)', 'orin' ),
			'section'	=> 'orin_single_pages_section',
		)
	);


	// Enable popup image for featured image and slider
	$wp_customize->add_setting(
		'orin_single_enable_popup_image',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_single_enable_popup_image',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Enable popup image for featured image and slider (Note: This setting applies to a single post page as well as a regular page)', 'orin' ),
			'section'	=> 'orin_single_pages_section',
		)
	);


	// Show tags
	$wp_customize->add_setting(
		'orin_single_show_tags',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_single_show_tags',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show tags', 'orin' ),
			'section'	=> 'orin_single_pages_section',
		)
	);


	// if "Orin Additional Features" plugin is activated
	if ( function_exists( 'orin_additional_features_show_share_buttons' ) ) {

		// Show social share buttons
		$wp_customize->add_setting(
			'orin_single_show_share_buttons',
			array(
				'default'			=> 1,
				'sanitize_callback'	=> 'orin_sanitize_checkbox',
			)
		);

		$wp_customize->add_control(
			'orin_single_show_share_buttons',
			array(
				'type'		=> 'checkbox',
				'label'		=> __( 'Show social share buttons (Twitter, Facebook, and Reddit)', 'orin' ),
				'section'	=> 'orin_single_pages_section',
			)
		);

	}


	// Show "About the author" section
	$wp_customize->add_setting(
		'orin_single_show_about_author',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_single_show_about_author',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show "About the author" section', 'orin' ),
			'section'	=> 'orin_single_pages_section',
		)
	);


	// Show "About the author" section: Description
	$wp_customize->add_setting(
		'orin_single_show_about_author_desc',
		array(
			'default'			=> __( 'You can add information about yourself in your profile settings: "WordPress Dashboard > Users > Profile > About Yourself > Biographical Info".', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_single_show_about_author_desc',
			array(
				'label'		=> __( 'Show "About the author" section (Setting Description)', 'orin' ),
				'section'	=> 'orin_single_pages_section',
				'settings'	=> 'orin_single_show_about_author_desc',
			)
		)
	);


	// Show post navigation
	$wp_customize->add_setting(
		'orin_single_show_post_navigation',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_single_show_post_navigation',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show post navigation', 'orin' ),
			'section'	=> 'orin_single_pages_section',
		)
	);


	// Show related posts
	$wp_customize->add_setting(
		'orin_single_show_related_posts',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_single_show_related_posts',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show related posts', 'orin' ),
			'section'	=> 'orin_single_pages_section',
		)
	);


	// Show related posts: Description
	$wp_customize->add_setting(
		'orin_single_show_related_posts_desc',
		array(
			'default'			=> __( 'This section displays posts that have at least one tag, which the current post also has. If the current post has no tags, then it has no related posts. If other posts do not have tags that the current post has, then the current post also has no related posts.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_single_show_related_posts_desc',
			array(
				'label'		=> __( 'Show related posts (Setting Description)', 'orin' ),
				'section'	=> 'orin_single_pages_section',
				'settings'	=> 'orin_single_show_related_posts_desc',
			)
		)
	);


	/**
	 * 5.2 - Single Pages: Regular Page
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_single_page_settings',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_single_page_settings',
			array(
				'label'		=> __( 'Regular Page', 'orin' ),
				'section'	=> 'orin_single_pages_section',
				'settings'	=> 'orin_single_page_settings',
			)
		)
	);


	// Show author
	$wp_customize->add_setting(
		'orin_page_show_author',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_page_show_author',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show author', 'orin' ),
			'section'	=> 'orin_single_pages_section',
		)
	);


	// Show date
	$wp_customize->add_setting(
		'orin_page_show_date',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_page_show_date',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show date', 'orin' ),
			'section'	=> 'orin_single_pages_section',
		)
	);


	// if "Orin Additional Features" plugin is activated
	if ( function_exists( 'orin_additional_features_show_share_buttons' ) ) {

		// Show social share buttons
		$wp_customize->add_setting(
			'orin_page_show_share_buttons',
			array(
				'default'			=> 1,
				'sanitize_callback'	=> 'orin_sanitize_checkbox',
			)
		);

		$wp_customize->add_control(
			'orin_page_show_share_buttons',
			array(
				'type'		=> 'checkbox',
				'label'		=> __( 'Show social share buttons (Twitter, Facebook, and Reddit)', 'orin' ),
				'section'	=> 'orin_single_pages_section',
			)
		);

	}


	// Show random posts at the bottom of the page
	$wp_customize->add_setting(
		'orin_page_show_random_posts',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_page_show_random_posts',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show random posts at the bottom of the page', 'orin' ),
			'section'	=> 'orin_single_pages_section',
		)
	);


	/**
	 * 6.0 - Footer
	 * -------------------------------------------------------------
	 */

	// Add new section
	$wp_customize->add_section(
		'orin_footer_section',
		array(
			'title'		=> __( 'Footer', 'orin' ),
			'priority'	=> 25,
		)
	);


	// Section description
	$wp_customize->add_setting(
		'orin_footer_section_desc',
		array(
			'default'			=> __( 'The footer area displays text and menu. The text usually contains copyright information. However, you can add your own links to this text, and you can also use icons (for example, social media icons). You can add links using the "a" tag, and you can add icons using the "i" tag. All available icons you can find on the Font Awesome site (A link to this site is available in the documentation, section Credits). To add a menu to the footer, go to the menu edit page - there you can create and add your menu to the footer (WordPress Dashboard > Appearance > Menus).', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_footer_section_desc',
			array(
				'label'		=> __( 'Footer: Description For The Settings Section', 'orin' ),
				'section'	=> 'orin_footer_section',
				'settings'	=> 'orin_footer_section_desc',
			)
		)
	);


	// Copyright text
	$wp_customize->add_setting(
		'orin_footer_text',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'orin_sanitize_wp_kses_text',
		)
	);

	$wp_customize->add_control(
		'orin_footer_text',
		array(
			'label'		=> __( 'Copyright Text', 'orin' ),
			'section'	=> 'orin_footer_section',
			'type'		=> 'textarea',
		)
	);


	// Allowed HTML tags (Copyright text)
	$wp_customize->add_setting(
		'orin_footer_text_desc',
		array(
			'default'			=> __( 'Allowed HTML tags: p, a, span, strong, b, em, i, br. Allowed attributes: p - class. a - href, title, target, class, rel. span - class. i - class.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_footer_text_desc',
			array(
				'label'		=> __( 'Copyright Text - Allowed HTML Tags', 'orin' ),
				'section'	=> 'orin_footer_section',
				'settings'	=> 'orin_footer_text_desc',
			)
		)
	);


	/**
	 * 7.0 - Styles And Cookies
	 * -------------------------------------------------------------
	 */

	// Add new section
	$wp_customize->add_section(
		'orin_colors_cookies_section',
		array(
			'title'		=> __( 'Styles And Cookies', 'orin' ),
			'priority'	=> 26,
		)
	);


	/**
	 * 7.1 - Styles And Cookies: Fonts And Color Mode
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_fonts_and_color_mode_settings',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_fonts_and_color_mode_settings',
			array(
				'label'		=> __( 'Fonts and Color Mode', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
				'settings'	=> 'orin_fonts_and_color_mode_settings',
			)
		)
	);


	// Default fonts: Sans-serif or serif
	$wp_customize->add_setting(
		'orin_default_font_type',
		array(
			'default'			=> 'sans-serif',
			'sanitize_callback'	=> 'orin_sanitize_font_type',
		)
	);

	$wp_customize->add_control(
		'orin_default_font_type',
		array(
			'type'		=> 'select',
			'label'		=> __( 'Default Fonts: Sans-Serif Or Serif', 'orin' ),
			'section'	=> 'orin_colors_cookies_section',
			'choices'	=> array(
				'sans-serif'	=> esc_html__( 'Sans-serif fonts', 'orin' ),
				'serif'			=> esc_html__( 'Serif fonts', 'orin' ),
			),
		)
	);


	// Default fonts: Description
	$wp_customize->add_setting(
		'orin_default_font_type_desc',
		array(
			'default'			=> __( 'This setting changes the default fonts (Sans-serif or Serif fonts). Important! If your current browser has the "orin_font_type" cookie, then the fonts will not change in that browser. The fonts will change for visitors who do not have this cookie.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_default_font_type_desc',
			array(
				'label'		=> __( 'Default Fonts (Setting Description)', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
				'settings'	=> 'orin_default_font_type_desc',
			)
		)
	);


	// Default site style: Light or dark
	$wp_customize->add_setting(
		'orin_default_theme_style',
		array(
			'default'			=> 'light',
			'sanitize_callback'	=> 'orin_sanitize_color_scheme',
		)
	);

	$wp_customize->add_control(
		'orin_default_theme_style',
		array(
			'type'		=> 'select',
			'label'		=> __( 'Default Site Style: Light Or Dark', 'orin' ),
			'section'	=> 'orin_colors_cookies_section',
			'choices'	=> array(
				'light'		=> esc_html__( 'Light', 'orin' ),
				'dark'		=> esc_html__( 'Dark', 'orin' ),
			),
		)
	);


	// Default site style: Description
	$wp_customize->add_setting(
		'orin_default_theme_style_desc',
		array(
			'default'			=> __( 'This setting changes the default color mode for your site (Light or Dark). Important! If your current browser has the "orin_site_style" cookie, then your site\'s color mode will not change in that browser. The color mode will change for visitors who do not have this cookie.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_default_theme_style_desc',
			array(
				'label'		=> __( 'Default Site Style (Setting Description)', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
				'settings'	=> 'orin_default_theme_style_desc',
			)
		)
	);


	/**
	 * 7.2 - Styles And Cookies: Colors
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_colors_settings',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_colors_settings',
			array(
				'label'		=> __( 'Colors', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
				'settings'	=> 'orin_colors_settings',
			)
		)
	);


	// Hover/active color (Light style)
	$wp_customize->add_setting(
		'orin_light_style_hover_color',
		array(
			'default'			=> '#6f6fff',
			'sanitize_callback'	=> 'sanitize_hex_color',
		)
	);

	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'orin_light_style_hover_color',
			array(
				'label'		=> esc_html__( 'Light Style: Hover/Active Color (Default Color: #6f6fff)', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
				'settings'	=> 'orin_light_style_hover_color',
			)
		)
	);


	// Hover/active color (Dark style)
	$wp_customize->add_setting(
		'orin_dark_style_hover_color',
		array(
			'default'			=> '#8585ff',
			'sanitize_callback'	=> 'sanitize_hex_color',
		)
	);

	$wp_customize->add_control(
		new WP_Customize_Color_Control(
			$wp_customize,
			'orin_dark_style_hover_color',
			array(
				'label'		=> esc_html__( 'Dark Style: Hover/Active Color (Default Color: #8585ff)', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
				'settings'	=> 'orin_dark_style_hover_color',
			)
		)
	);


	/**
	 * 7.3 - Styles And Cookies: Cookies
	 * -------------------------------------------------------------
	 */

	// Heading
	$wp_customize->add_setting(
		'orin_cookies_settings',
		array(
			'default'			=> '',
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_heading(
			$wp_customize,
			'orin_cookies_settings',
			array(
				'label'		=> __( 'Cookies', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
				'settings'	=> 'orin_cookies_settings',
			)
		)
	);


	// Information about cookies
	$wp_customize->add_setting(
		'orin_about_cookies',
		array(
			'default'			=> __( 'HTTP cookies (also called web cookies, Internet cookies, browser cookies, or simply cookies) are small blocks of data created by a web server while a user is browsing a website and placed on the user\'s computer or other device by the user\'s web browser. This theme uses cookies to store the theme style (light or dark) and the selected font type (sans-serif or serif).', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_about_cookies',
			array(
				'label'		=> __( 'Information About Cookies', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
				'settings'	=> 'orin_about_cookies',
			)
		)
	);


	// Enable cookies for the font switching function
	$wp_customize->add_setting(
		'orin_font_switch_enable_cookies',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_font_switch_enable_cookies',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Use cookies to save the selected type of fonts (sans-serif or serif fonts)', 'orin' ),
			'section'	=> 'orin_colors_cookies_section',
		)
	);


	// Enable cookies for the theme color switching function
	$wp_customize->add_setting(
		'orin_color_switch_enable_cookies',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_color_switch_enable_cookies',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Use cookies to save the selected site style (light or dark style)', 'orin' ),
			'section'	=> 'orin_colors_cookies_section',
		)
	);


	// if "Orin Additional Features" plugin is activated
	if ( function_exists( 'orin_additional_features_show_cookies_notice' ) ) {

		// Show cookies notice
		$wp_customize->add_setting(
			'orin_show_cookies_notice',
			array(
				'default'			=> 0,
				'sanitize_callback'	=> 'orin_sanitize_checkbox',
			)
		);

		$wp_customize->add_control(
			'orin_show_cookies_notice',
			array(
				'type'		=> 'checkbox',
				'label'		=> __( 'Show cookies notice', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
			)
		);


		// Cookies notice on mobile devices
		$wp_customize->add_setting(
			'orin_cookies_notice_on_mobile',
			array(
				'default'			=> 'hidden',
				'sanitize_callback'	=> 'orin_sanitize_cookies_notice_on_mobile',
			)
		);

		$wp_customize->add_control(
			'orin_cookies_notice_on_mobile',
			array(
				'type'		=> 'select',
				'label'		=> __( 'Cookies Notice On Mobile Devices', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
				'choices'	=> array(
					'hidden'	=> esc_html__( 'Hidden notification', 'orin' ),
					'visible'	=> esc_html__( 'Visible notification', 'orin' ),
				),
			)
		);


		// Cookies notice: Title
		$wp_customize->add_setting(
			'orin_cookies_notice_title',
			array(
				'default'			=> 'Cookies Notice',
				'sanitize_callback'	=> 'esc_html',
			)
		);

		$wp_customize->add_control(
			'orin_cookies_notice_title',
			array(
				'label'		=> __( 'Cookies Notice: Title (Required Field)', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
				'type'		=> 'text',
			)
		);


		// Cookies notice: Text
		$wp_customize->add_setting(
			'orin_cookies_notice_text',
			array(
				'default'			=> 'Our website use cookies. If you continue to use this site we will assume that you are happy with this.',
				'sanitize_callback'	=> 'orin_sanitize_wp_kses_text',
			)
		);

		$wp_customize->add_control(
			'orin_cookies_notice_text',
			array(
				'label'		=> __( 'Cookies Notice: Text (Required Field)', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
				'type'		=> 'textarea',
			)
		);


		// Allowed HTML tags (Cookies notice: Text)
		$wp_customize->add_setting(
			'orin_cookies_notice_text_desc',
			array(
				'default'			=> __( 'Allowed HTML tags: a, span, strong, b, em, i. Allowed attributes: a - href, title, target, class, rel. span - class. i - class.', 'orin' ),
				'sanitize_callback'	=> 'esc_html',
			)
		);

		$wp_customize->add_control(
			new orin_customize_control_description(
				$wp_customize,
				'orin_cookies_notice_text_desc',
				array(
					'label'		=> __( 'Cookies Notice: Text - Allowed HTML Tags', 'orin' ),
					'section'	=> 'orin_colors_cookies_section',
					'settings'	=> 'orin_cookies_notice_text_desc',
				)
			)
		);


		// Cookies notice: Button text
		$wp_customize->add_setting(
			'orin_cookies_notice_button_text',
			array(
				'default'			=> 'I Agree',
				'sanitize_callback'	=> 'esc_html',
			)
		);

		$wp_customize->add_control(
			'orin_cookies_notice_button_text',
			array(
				'label'		=> __( 'Cookies Notice: Button Text (Required Field)', 'orin' ),
				'section'	=> 'orin_colors_cookies_section',
				'type'		=> 'text',
			)
		);

	}


	/**
	 * 8.0 - Other Theme Settings
	 * -------------------------------------------------------------
	 */

	// Add new section
	$wp_customize->add_section(
		'orin_other_settings_section',
		array(
			'title'		=> __( 'Other Theme Settings', 'orin' ),
			'priority'	=> 27,
		)
	);


	// Section description
	$wp_customize->add_setting(
		'orin_other_settings_section_desc',
		array(
			'default'			=> __( 'Section with additional settings for the Orin theme.', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_other_settings_section_desc',
			array(
				'label'		=> __( 'Description for the Other Settings section', 'orin' ),
				'section'	=> 'orin_other_settings_section',
				'settings'	=> 'orin_other_settings_section_desc',
			)
		)
	);


	// Show "Back to top" button
	$wp_customize->add_setting(
		'orin_show_back_to_top',
		array(
			'default'			=> 1,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_show_back_to_top',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Show "Back to top" button (Note: The button is always hidden on mobile devices)', 'orin' ),
			'section'	=> 'orin_other_settings_section',
		)
	);


	// Cancel text transformation for some text elements
	$wp_customize->add_setting(
		'orin_cancel_text_transform',
		array(
			'default'			=> 0,
			'sanitize_callback'	=> 'orin_sanitize_checkbox',
		)
	);

	$wp_customize->add_control(
		'orin_cancel_text_transform',
		array(
			'type'		=> 'checkbox',
			'label'		=> __( 'Cancel text transformation for some text elements', 'orin' ),
			'section'	=> 'orin_other_settings_section',
		)
	);


	// Cancel text transformation for some text elements: Description
	$wp_customize->add_setting(
		'orin_cancel_text_transform_desc',
		array(
			'default'			=> __( 'This setting removes the "text-transform: capitalize" property for some text elements. Example: "You May Also Like" > "You may also like".', 'orin' ),
			'sanitize_callback'	=> 'esc_html',
		)
	);

	$wp_customize->add_control(
		new orin_customize_control_description(
			$wp_customize,
			'orin_cancel_text_transform_desc',
			array(
				'label'		=> __( 'Cancel text transformation for some text elements (Setting Description)', 'orin' ),
				'section'	=> 'orin_other_settings_section',
				'settings'	=> 'orin_cancel_text_transform_desc',
			)
		)
	);

}
add_action( 'customize_register', 'orin_customize_register' );
